/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.device;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum VehicleFeatureCategory implements HasName {

  // External device detected

  /**
   * An unknown device.
   */
  @JsonEnumDefaultValue
  UNKNOWN_DEVICE("UnknownDevice", 0),

  /**
   * Third-party Garmin device.
   */
  GARMIN("Garmin", 1),

  /**
   * Third-party Iridium device.
   */
  IRIDIUM("Iridium", 2),

  /**
   * Legacy hours of service device.
   */
  HOS("Hos", 3),

  /**
   * External NFC device.
   */
  NFC("Nfc", 4),

  /**
   * External GoTalk device.
   */
  GO_TALK("GoTalk", 5),

  /**
   * Third-party Mobileye device.
   */
  MOBILEYE("Mobileye", 6),

  /**
   * Third-party Valor device.
   */
  VALOR("Valor", 7),

  /**
   * Third-party Valor device.
   */
  WIFI("WiFi", 8),

  /**
   * Third-party salt spreader device.
   */
  SALT_SPREADER("SaltSpreader", 9),

  /**
   * Active tracking.
   */
  GO_ACTIVE("GoActive", 10),

  /**
   * OBD Alert setting present.
   */
  OBD_PRESENT("OBDPresent", 11),

  /**
   * OBD Alert setting enabled.
   */
  OBD_ENABLED("OBDEnabled", 12),

  // Manual activation

  /**
   * Garmin hours of service.
   */
  GARMIN_HOS("GarminHos", 1001),

  /**
   * Geotab Drive hours of service.
   */
  GEOTAB_DRIVE_HOS("GeotabDriveHos", 1002);

  private final String name;
  private final int code;

  VehicleFeatureCategory(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }


  public static VehicleFeatureCategory findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          VehicleFeatureCategory.class.getSimpleName(), VehicleFeatureCategory.UNKNOWN_DEVICE);
      return UNKNOWN_DEVICE;
    }

    for (VehicleFeatureCategory vehicleFeatureCategory : values()) {
      if (vehicleFeatureCategory.getName().equalsIgnoreCase(name.trim())) {
        return vehicleFeatureCategory;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, VehicleFeatureCategory.class.getSimpleName(), VehicleFeatureCategory.UNKNOWN_DEVICE);
    return UNKNOWN_DEVICE;
  }
}
