/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.devicestatusinfo;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.exceptionevent.ExceptionEvent;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.trip.Trip;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.DutyStatusInfoDriverSerializer;
import com.geotab.model.serialization.EntityCollectionAsIdCollectionSerializer;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Represents the current state of a vehicle by providing information such as the vehicle bearing location and speed,
 * active exception events and whether the device is currently communicating.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class DeviceStatusInfo extends Entity {

  /**
   * The duration between the last Trip state change (i.e. driving or stop), and the most recent date of location
   * information.
   */
  private Duration currentStateDuration;

  /**
   * The bearing (heading) in integer degrees.
   *
   * <p>Valued between 0 and 359 inclusive. 0 represents North, 90 represents East, and so on. -1
   * represents unknown bearing.
   */
  private Integer bearing;

  /**
   * The {@link ExceptionEvent}(s) that are currently active.
   */
  private List<ExceptionEvent> exceptionEvents;

  /**
   * A value indicating whether the {@link Device} is communicating.
   */
  private Boolean isDeviceCommunicating;

  /**
   * A value indicating whether the current {@link Device} state. If set true, is driving. Otherwise, it is stopped.
   */
  private Boolean isDriving;

  /**
   * The current latitude of the {@link Device}.isEmailReportEnabled
   */
  private Double latitude;

  /**
   * The current longitude of the {@link Device}.
   */
  private Double longitude;

  /**
   * The current vehicle speed.
   *
   * <p>NaN represents an invalid speed.
   */
  private Float speed;

  /**
   * The most recent dateTime of the latest piece of status, gps or fault data.
   */
  private LocalDateTime dateTime;

  /**
   * The {@link Device} this {@link DeviceStatusInfo} belongs to.
   */
  private Device device;

  /**
   * The {@link User} driver associated to the current {@link Device}.
   */
  @JsonSerialize(using = DutyStatusInfoDriverSerializer.class)
  private Driver driver;

  /**
   * A value indicating whether the {@link Device} has been assigned to "UnknownDriver" and the last {@link Trip}{@link
   * Driver} is returned.
   *
   * <p>[true] while the {@link Device} is assigned to "UnknownDriver" and no other trip had been
   * made in the vehicle and/or the last {@link Trip}{@link Driver} has not made a trip in another vehicle, otherwise
   * [false].
   */
  private Boolean isHistoricLastDriver;

  /**
   * The {@link Group}(s) that the {@link Device} currently belongs to.
   */
  @JsonSerialize(converter = EntityCollectionAsIdCollectionSerializer.class)
  private List<Group> groups;
}
