/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.diagnostic;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.entity.NameEntityWithVersion;
import com.geotab.model.entity.controller.Controller;
import com.geotab.model.entity.enginetype.EngineType;
import com.geotab.model.entity.faultdata.FaultResetMode;
import com.geotab.model.entity.source.Source;
import com.geotab.model.entity.unitofmeasure.UnitOfMeasure;
import com.geotab.model.serialization.DiagnosticDeserializer;
import com.geotab.model.serialization.EntityAsIdSerializer;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Vehicle diagnostic information from the engine computer that can either be measurement data or fault code data.
 *
 * <p>Note: Diagnostics cannot be added, set or removed via the API..
 */
@Data
@NoArgsConstructor
@SuperBuilder
@JsonDeserialize(using = DiagnosticDeserializer.class)
public abstract class Diagnostic extends NameEntityWithVersion {

  public static final List<BasicDiagnostic> TAMPERING_DIAGNOSTICS = Stream.of(
          "DiagnosticDeviceRestartedBecauseAllPowerWasRemovedId",
          "DiagnosticGpsAntennaUnpluggedId",
          "DiagnosticGpsAntennaShortCircuitId")
      .map(id -> {
        BasicDiagnostic out = new BasicDiagnostic();
        out.setId(new Id(id));
        return out;
      }).collect(Collectors.toList());

  /**
   * The diagnostic parameter code number.
   */
  protected Integer code;

  /**
   * The applicable {@link Controller} for the diagnostic parameter.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  protected Controller controller;

  /**
   * The {@link DiagnosticType} (source) of the diagnostic parameter.
   */
  protected DiagnosticType diagnosticType;

  /**
   * The {@link EngineType} for this diagnostic if applicable.
   */
  protected EngineType engineType;

  /**
   * The {@link FaultResetMode} of the diagnostic (whether the fault resets automatically or manually).
   */
  protected FaultResetMode faultResetMode;

  /**
   * The {@link Source} for the diagnostic (the type of diagnostic code).
   */
  protected Source source;

  /**
   * The {@link UnitOfMeasure} applicable to the diagnostic parameter.
   */
  protected UnitOfMeasure unitOfMeasure;

  /**
   * The {@link ValidLoggingPeriod}. The diagnostic should only be interpolated within the range of the given type.
   */
  protected ValidLoggingPeriod validLoggingPeriod;

  /**
   * The value which describes if the diagnostic guarantees it will log when estimate error is exceeded.
   */
  @JsonProperty("isLogGuaranteedOnEstimateError")
  protected Boolean isLogGuaranteedOnEstimateError;
}
