/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.dutystatus;

import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.address.AddressLookupData;
import com.geotab.model.entity.EntityWithVersion;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.hos.UserHosRuleSet;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.EntityAsIdSerializer;
import com.geotab.model.serialization.EntityCollectionAsIdCollectionSerializer;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * Driver Availability for Hours of Service regulations.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class DutyStatusLog extends EntityWithVersion {

  /**
   * The event type number of this log 1 = A change in driver's duty-status 2 = An intermediate log 3 = A change in
   * driver's indication of authorized personal use of CMV or yard moves 4 = A driver's certification/re-certification
   * of records 5 = A driver's login/logout activity 6 = CMV's engine power up / shut down activity 7 = A malfunction or
   * data diagnostic detection occurrence (Table 6; 7.25 of the ELD Final Rule).
   */
  private Byte eventType;

  /**
   * The event code of this log (Table 6; 7.20 of the ELD Final Rule).
   */
  private Byte eventCode;

  /**
   * The record status number of this log. 1 = active 2 = inactive - changed 3 = inactive - change requested 4 =
   * inactive - change rejected.
   */
  private Byte eventRecordStatus;

  /**
   * The {@link Device} associated with this log.
   */
  private Device device;

  /**
   * The User who created the log.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User driver;

  /**
   * The date and time the log was created.
   */
  private LocalDateTime dateTime;

  /**
   * The {@link DutyStatusLogType} representing the driver's duty status.
   */
  private DutyStatusLogType status;

  /**
   * The date and time the log was verified.
   */
  private LocalDateTime verifyDateTime;

  /**
   * The date and time the log was edited.
   */
  private LocalDateTime editDateTime;

  /**
   * The list of the co-driver {@link User}(s) for this log.
   */
  @JsonSerialize(converter = EntityCollectionAsIdCollectionSerializer.class)
  private List<User> coDrivers;

  /**
   * The list of {@link AnnotationLog}(s) which are associated with this log.
   */
  private List<AnnotationLog> annotations;

  /**
   * The location data for this  log.
   */
  private AddressLookupData location;

  /**
   * The {@link DutyStatusOrigin} data for this log.
   */
  private DutyStatusOrigin origin;

  /**
   * The id of the first version of this log.
   */
  private Id parentId;

  /**
   * The record status of the log record.
   */
  private DutyStatusState state;

  /**
   * The sequence number the log record.
   */
  private Long sequence;

  /**
   * The {@link DutyStatusMalfunctionTypes} of this log.
   */
  private DutyStatusMalfunctionTypes malfunction;

  /**
   * The engine hours at the time of the log record.
   */
  private Double engineHours;

  /**
   * The vehicle's odometer at the time of the log record.
   */
  private Double odometer;

  /**
   * The distance travelled since the last valid coordinate before the log's datetime.
   */
  private Double distanceSinceValidCoordinates;

  /**
   * The deferral status.
   */
  private DutyStatusDeferralType deferralStatus;

  /**
   * The deferral minutes.
   */
  private Integer deferralMinutes;

  /**
   * Is the log ignored.
   */
  private Boolean isIgnored;

  /**
   * Event checksum of a duty status log according to ELD specifications.
   */
  private Long eventCheckSum;

  /**
   * The {@link User} that requested an edit to this log.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User editRequestedByUser;

  /**
   * Hides the logs from the drivers on the drive app.
   */
  private Boolean isHidden;

  /**
   * The linked ruleset for canadian compliance logs.
   */
  private UserHosRuleSet userHosRuleSet;

  /**
   * If log is in the Transitioning state.
   */
  private Boolean isTransitioning;
}
