/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.dvirlog;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.address.AddressLookupData;
import com.geotab.model.entity.NameEntityWithVersion;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.trailer.Trailer;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.DefectListDeserializer;
import com.geotab.model.serialization.EntityAsIdSerializer;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A DVIRLog is a Driver Vehicle Inspection Report which is prepared by a driver regarding defects in parts of a vehicle
 * associated with a {@link Device} or {@link Trailer}.
 */
@Data
@NoArgsConstructor
@SuperBuilder
@SuppressWarnings("AbbreviationAsWordInName")
public class DVIRLog extends NameEntityWithVersion {

  /**
   * The {@link User} who certified the repairs (or comments if no repairs were made) to the Device or Trailer.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User certifiedBy;

  /**
   * The date the Device or Trailer was certified.
   */
  private LocalDateTime certifyDate;

  /**
   * The remark recorded by the {@link User} who certified the repairs (or no repairs made) to the Device or Trailer.
   */
  private String certifyRemark;

  /**
   * The list of defect {@link Group}(s) for this log.
   */
  @JsonDeserialize(using = DefectListDeserializer.class)
  private List<Group> defects;

  /**
   * The Device associated with this log. Either a Device or a Trailer is defined for a log, not both (if the device is
   * set, trailer must be null).
   */
  private Device device;

  /**
   * The list of DVIRDefects{@link DVIRDefect}(s) for this log.
   */
  @JsonProperty("dVIRDefects")
  @SuppressWarnings({"AbbreviationAsWordInName"})
  private List<DVIRDefect> dVIRDefects;

  /**
   * The remark recorded by the driver for this log.
   */
  private String driverRemark;

  /**
   * The identifier for whether or not the Device or Trailer was certified as safe to operate.
   */
  @JsonProperty("isSafeToOperate")
  private Boolean isSafeToOperate;

  /**
   * The date the Device or Trailer was repaired.
   */
  private LocalDateTime repairDate;

  /**
   * The User who repaired the Device or Trailer.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User repairedBy;

  /**
   * The remark recorded by the User who repaired the Device or Trailer.
   */
  private String repairRemark;

  /**
   * The Trailer associated with this log. Either a Device or a Trailer is defined for a log, not both (if the device is
   * set, trailer must be null).
   */
  private Trailer trailer;

  /**
   * The date and time the log was created.
   */
  private LocalDateTime dateTime;

  /**
   * The User who created the log.
   */
  @JsonSerialize(using = EntityAsIdSerializer.class)
  private User driver;

  /**
   * The {@link DVIRLogType} of the log. Default [Unknown].
   */
  private DVIRLogType logType;

  /**
   * An object with the location information of the log.
   */
  private AddressLookupData location;

  /**
   * The defect list {@link Group} of the log.
   */
  private Group defectList;

  /**
   * The authority name for the driver at the time of this log. Maximum length [255] Default [""].
   */
  private String authorityName;

  /**
   * The authority address for the driver at the time of this log. Maximum length [255] Default [""].
   */
  private String authorityAddress;

  /**
   * The odometer or hubometer of the vehicle or trailer, respectively, if it was manually recorded by the driver.
   * Default [null].
   */
  private Double odometer;

  /**
   * The load height, if it was manually recorded by the driver. Default [null].
   */
  private Float loadHeight;

  /**
   * The load width, if it was manually recorded by the driver. Default [null].
   */
  private Float loadWidth;

  /**
   * A value indicating whether the DVIR log was inspected by the driver. If false, the log was inspected by an
   * inspector. Default [true].
   */
  @JsonProperty("isInspectedByDriver")
  private Boolean isInspectedByDriver;

  /**
   * A value indicating whether the DVIR log was marked as rejected by the driver. Default [false].
   */
  @JsonProperty("isRejected")
  private Boolean isRejected;
}
