/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.group;

import static com.fasterxml.jackson.annotation.JsonTypeInfo.As.EXISTING_PROPERTY;
import static com.fasterxml.jackson.annotation.JsonTypeInfo.Id.NAME;
import static com.geotab.model.entity.group.CompanyGroup.COMPANY_GROUP_ID;
import static com.geotab.model.entity.group.DefectsGroup.DEFECTS_GROUP_ID;
import static com.geotab.model.entity.group.DriveUserSecurityGroup.DRIVE_USER_SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.EverythingSecurityGroup.EVERYTHING_SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.NothingSecurityGroup.NOTHING_SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.PrivateUserGroup.PRIVATE_USER_GROUP_ID;
import static com.geotab.model.entity.group.RootGroup.ROOT_GROUP_ID;
import static com.geotab.model.entity.group.SecurityGroup.SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.SupervisorSecurityGroup.SUPERVISOR_SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.UserSecurityGroup.USER_SECURITY_GROUP_ID;
import static com.geotab.model.entity.group.ViewOnlySecurityGroup.VIEW_ONLY_SECURITY_GROUP_ID;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonSubTypes.Type;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.geotab.model.drawing.Color;
import com.geotab.model.entity.NameEntity;
import java.util.List;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * A group is one element in a hierarchical tree. Each group can have none or many children, it is the children that
 * define the shape of the  hierarchical tree. The parent is not a property of the group object and is only defined by
 * who the group is a child of. It is necessary to know the id of the parent group when adding a new group or removing
 * an existing group.
 *
 * <p>There are three core Group branches used in MyGeotab.
 *
 * <p>The most common are "Company Groups",
 * company Groups are used to organize entities Zone, User, Device, Driver and Exception Rules into logical groups
 * related to the organization. A Group structure can be constructed by region, vocation, reporting or anything that
 * makes sense to the business, this allows aggregate reports and rolling up data in a flexible way. These groups have a
 * many to many type of relationship with the entities that are members and are not limited to one type of entity.
 *
 * <p>The second type is "Security Groups", these are Groups to which User(s) are
 * members of and can only be applied to Users. Each Group has a list of SecurityFilter(s) associated to it. Security
 * Filters control what parts of the application/API a User has access to.
 *
 * <p>The third type of group is a "Private User Group", this group is used only for scheduling ts
 * and displaying dashboard reports for a User. This Group will only ever apply to one User ill typically be named the
 * user's name.
 *
 * <p>There is a base structure of Groups which cannot be removed, these are considered to be
 * "System"
 * <ul>
 *     <li>
 *         {@link RootGroup}
 *    </li>
 *    <li>
 *         {@link CompanyGroup}
 *    </li>
 *     <li>
 *         {@link SecurityGroup}
 *     </li>
 *     <li>
 *         {@link EverythingSecurityGroup}
 *     </li>
 *     <li>
 *         {@link SupervisorSecurityGroup}
 *     </li>
 *     <li>
 *         {@link ViewOnlySecurityGroup}
 *     </li>
 *     <li>
 *         {@link NothingSecurityGroup}
 *     </li>
 *     <li>
 *        {@link UserSecurityGroup}
 *     </li>
 *     <li>
 *        {@link DriveUserSecurityGroup}
 *     </li>
 *     <li>
 *         {@link PrivateUserGroup}
 *     </li>
 *     <li>
 *         {@link DefectsGroup}
 *     </li>
 * </ul>
 *
 * <p>When Groups are retrieved they will always be in a flat list of groups. The hierarchically
 * tree(s) can be reconstructed by looking at the "Children" property of each Group.
 * The "Root" group will never be returned and is only for system use.
 */
@Data
@NoArgsConstructor
@SuperBuilder
@JsonTypeInfo(use = NAME, include = EXISTING_PROPERTY, property = "id", visible = true, defaultImpl = Group.class)
@JsonSubTypes({
    @Type(value = RootGroup.class, name = ROOT_GROUP_ID),
    @Type(value = CompanyGroup.class, name = COMPANY_GROUP_ID),
    @Type(value = SecurityGroup.class, name = SECURITY_GROUP_ID),
    @Type(value = EverythingSecurityGroup.class, name = EVERYTHING_SECURITY_GROUP_ID),
    @Type(value = SupervisorSecurityGroup.class, name = SUPERVISOR_SECURITY_GROUP_ID),
    @Type(value = ViewOnlySecurityGroup.class, name = VIEW_ONLY_SECURITY_GROUP_ID),
    @Type(value = NothingSecurityGroup.class, name = NOTHING_SECURITY_GROUP_ID),
    @Type(value = UserSecurityGroup.class, name = USER_SECURITY_GROUP_ID),
    @Type(value = DriveUserSecurityGroup.class, name = DRIVE_USER_SECURITY_GROUP_ID),
    @Type(value = PrivateUserGroup.class, name = PRIVATE_USER_GROUP_ID),
    @Type(value = DefectsGroup.class, name = DEFECTS_GROUP_ID)})
public class Group extends NameEntity {
  // NOTE: be aware of the custom GroupSerializer

  /**
   * The {@link Color} is used to render assets belonging to this group. Default [Blue].
   */
  private Color color;

  /**
   * The parent Group of the selected group. Not defined in the group object, the relationship is assumed by which group
   * has this group as a child.
   */
  private Group parent;

  /**
   * Children of this group. A list of Group(s).
   */
  private List<Group> children;

  /**
   * Free text field where any user information can be stored and referenced for this entity. Default [""].
   */
  private String comments;

  /**
   * The string reference to add to the database entry for this group. Maximum length [255] Default [""].
   */
  private String reference;

  @JsonProperty("isGlobalReportingGroup")
  private Boolean isGlobalReportingGroup;
}
