/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.route;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.entity.device.Device;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum RouteType implements HasName {

  /**
   * A basic route. Does not have a {@link Device} associated.
   */
  @JsonEnumDefaultValue
  BASIC("Basic", 0),

  /**
   * A route with a plan of the time/distance it is expected to take from stop to stop. A plan route has a {@link
   * Device} associated.
   */
  PLAN("Plan", 1),

  /**
   * A route with a plan with completion criteria. A service route has a {@link Device} associated.
   */
  SERVICE("Service", 2);

  private final String name;
  private final int code;

  RouteType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static RouteType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          RouteType.class.getSimpleName(), RouteType.BASIC);
      return BASIC;
    }

    for (RouteType routeType : values()) {
      if (routeType.getName().equalsIgnoreCase(name.trim())) {
        return routeType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, RouteType.class.getSimpleName(), RouteType.BASIC);
    return BASIC;
  }
}

