package com.geotab.model.entity.tachograph.files;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.Id;
import com.geotab.model.entity.EntityWithVersion;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.entitysettings.TachographFileFormat;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.serialization.ByteArrayDeserializer;
import com.geotab.model.serialization.ByteArraySerializer;
import java.time.LocalDateTime;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * The entity which describes the tachograph data file.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class TachographDataFile extends EntityWithVersion {
  /**
   * Gets or sets the {@link Driver} related with the file data.
   */
  private Driver driver;

  /**
   * Gets or sets the {@link Device} related with the file data.
   */
  private Device device;

  /**
   * Gets or sets the associated binary data object.
   */
  @JsonDeserialize(using = ByteArrayDeserializer.class)
  @JsonSerialize(using = ByteArraySerializer.class)
  private Byte[] binaryData;

  /**
   * Gets or sets the origin of the file.
   */
  private int origin;

  /**
   * Gets or sets the description of the origin of the file.
   */
  private String originDescription;

  /**
   * Gets or sets the date on which the file was uploaded.
   */
  private LocalDateTime dateTime;

  /**
   * Gets or sets the date on which the file was uploaded.
   */
  private LocalDateTime uploadDateTime;

  /**
   * Gets or sets the date on which the file was generated by a tachograph or reader.
   */
  private LocalDateTime timestamp;

  /**
   * Gets or sets the minimum date containing all data.
   */
  private LocalDateTime timestampStart;

  /**
   * Gets or sets the most recent date with activity.
   */
  private LocalDateTime timestampEnd;

  /**
   * Gets or sets the file name.
   */
  private String fileName;

  /**
   * Gets or sets the summary about the information of the file.
   */
  private String summary;

  /**
   * Gets or sets the signature.
   */
  private int signature;

  /**
   * Gets or sets the TGD file name.
   */
  private String fileNameTgd;

  /**
   * Gets or sets the DDD file name.
   */
  private String fileNameDdd;

  /**
   * Gets or sets the V1B file name.
   */
  private String fileNameV1B;

  /**
   * Gets or sets the String with the warnings found in the download, the download is correct for the administration.
   */
  private String warnings;

  /**
   * Gets or sets the String with the errors found in the download, the download is NOT valid and will have to be repeated.
   */
  private String errors;

  /**
   * Gets or sets the int code for the processing status.
   */
  private int processingStatus;

  /**
   * Gets or sets the value that contains file gaps.
   */
  private String lockGaps;

  /**
   * Gets or sets the value that indicates whether it's archived.
   */
  private Boolean archived;

  /**
   * Initializes a new instance of the {@link TachographDataFile} class.
   *
   * @param value The value.
   */
  public TachographDataFile(TachographDataFile value) {
    this(value.getId(), value.getDriver(), value.getDevice(), value.getBinaryData(), value.getOrigin(), value.getOriginDescription(),
        value.getDateTime(), value.getTimestamp(), value.getTimestampStart(), value.getTimestampEnd(), value.getFileName(), value.getSummary(),
        value.getSignature(), value.getFileNameTgd(), value.getFileNameDdd(), value.getFileNameV1B(), value.getWarnings(), value.getErrors(),
        value.getProcessingStatus(), value.getLockGaps(), value.getArchived());
  }

  /**
   * Initializes a new instance of the {@link TachographDataFile}.
   *
   * @param id The {@link Id}.
   * @param driver The driver.
   * @param device The vehicle.
   * @param binaryData The file binary data.
   * @param origin The origin of the file.
   * @param originDescription The description of the origin of the file.
   * @param dateTime The date on which the file was uploaded.
   * @param timestamp The date on which the file was generated by a tachograph or reader.
   * @param timestampStart The minimum date containing all data.
   * @param timestampEnd The most recent date with activity.
   * @param fileName The file name.
   * @param summary The summary.
   * @param signature The signature.
   * @param fileNameTgd The TGD file name.
   * @param fileNameDdd The DDD file name.
   * @param fileNameV1B The V1B file name.
   * @param warnings The String with the warnings found in the download, the download is correct for the administration.
   * @param errors The String with the errors found in the download, the download is NOT valid and will have to be repeated.
   * @param processingStatus The processing status.
   * @param lockGaps The String value of file gaps.
   * @param archived The boolean value of archived.
   */
  public TachographDataFile(Id id, Driver driver, Device device, Byte[] binaryData, int origin, String originDescription, LocalDateTime dateTime,
      LocalDateTime timestamp, LocalDateTime timestampStart, LocalDateTime timestampEnd, String fileName, String summary, int signature,
      String fileNameTgd,
      String fileNameDdd, String fileNameV1B, String warnings, String errors, int processingStatus, String lockGaps, Boolean archived) {
    this.setId(id);
    this.driver = driver;
    this.device = device;
    this.binaryData = binaryData;
    this.origin = origin;
    this.originDescription = originDescription;
    this.dateTime = dateTime;
    this.timestamp = timestamp;
    this.timestampStart = timestampStart;
    this.timestampEnd = timestampEnd;
    this.fileName = fileName;
    this.summary = summary;
    this.signature = signature;
    this.fileNameTgd = fileNameTgd;
    this.fileNameDdd = fileNameDdd;
    this.fileNameV1B = fileNameV1B;
    this.warnings = warnings;
    this.errors = errors;
    this.processingStatus = processingStatus;
    this.lockGaps = lockGaps;
    this.archived = archived;
  }

  /**
   * Get FileName from TachographFileFormat or TGD as default.
   *
   * @param tachographFileFormat The file format to get the file name from.
   * @return Get Filename from a Tachograph file format or default File format TGD.
   */
  public String getFileNameFromFileFormatOrDefault(TachographFileFormat tachographFileFormat) {
    switch (tachographFileFormat) {
      case Ddd:
        return this.fileNameDdd;
      case V1b:
        return this.fileNameV1B;
      case Tgd:
      default:
        return this.fileNameTgd;
    }
  }
}
