package com.geotab.model.entity.textmessage;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.entity.device.GoDevice;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum MessageContentType implements HasName {

  /**
   * Basic text message.
   */
  @JsonEnumDefaultValue
  NORMAL(TextMessageContentType.NORMAL, 0),

  /**
   * Text message that also includes response options.
   */
  CANNED_RESPONSE(TextMessageContentType.CANNED_RESPONSE, 1),

  /**
   * Text message that includes a location.
   */
  LOCATION(TextMessageContentType.LOCATION, 2),

  /**
   * Content Type for sending data to a component.
   */
  DATA_TO_COMPONENT(TextMessageContentType.DATA_TO_COMPONENT, 3),

  /**
   * Content Type for a status data request.
   */
  STATUS_DATA_REQUEST(TextMessageContentType.STATUS_DATA_REQUEST, 4),

  /**
   * Content Type for immobilization.
   */
  IMMOBILIZATION(TextMessageContentType.IMMOBILIZATION, 5),

  /**
   * Text message that is converted to speech.  Must have GOTalk.
   */
  GO_TALK(TextMessageContentType.GO_TALK, 6),

  /**
   * Text message that includes instructions to open or close an IOX-OUTPUT relay.
   */
  IOX_OUTPUT(TextMessageContentType.IOX_OUTPUT, 7),

  /**
   * Text message that causes the GoDevice to beep.
   */
  BEEP(TextMessageContentType.BEEP, 8),

  /**
   * Text message that contains bytes to give to a third-party IOX Add-On.
   */
  SERIAL_IOX(TextMessageContentType.SERIAL_IOX, 9),

  /**
   * Text message with information to add/remove a driver from a {@link GoDevice}'s white list.
   */
  DRIVER_WHITE_LIST(TextMessageContentType.DRIVER_WHITE_LIST, 10),

  /**
   * The MIME content.
   */
  MIME_CONTENT(TextMessageContentType.MIME_CONTENT, 11),

  /**
   * Text message that contains bytes to give to a third-party IOX Add-On.
   */
  CAN(TextMessageContentType.CAN, 12),

  /**
   * Text message that contains bytes to give to a third-party IOX Add-On.
   */
  PASSTHROUGH_ENABLED_MESSAGE(TextMessageContentType.PASSTHROUGH_ENABLED_MESSAGE, 13),

  /**
   * Content used to issue commands to a vehicle (for example locking doors).
   */
  VEHICLE_COMMAND(TextMessageContentType.VEHICLE_COMMAND, 14),

  OTAU_CONTENT(TextMessageContentType.OTAU_CONTENT, 15),

  WORKSHOP_MODE(TextMessageContentType.WORKSHOP_MODE, 16),

  FLASH_MEMORY(TextMessageContentType.FLASH_MEMORY, 17),

  MIME_REFERENCE_CONTENT(TextMessageContentType.MIME_REFERENCE_CONTENT, 18);

  private final String name;
  private final int code;

  MessageContentType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static MessageContentType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          MessageContentType.class.getSimpleName(), MessageContentType.NORMAL);
      return NORMAL;
    }

    for (MessageContentType authenticationType : values()) {
      if (authenticationType.getName().equalsIgnoreCase(name.trim())) {
        return authenticationType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, MessageContentType.class.getSimpleName(), MessageContentType.NORMAL);
    return NORMAL;
  }

  private static class Constants {

  }
}
