/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.textmessage;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.serialization.ByteArrayDeserializer;
import com.geotab.model.serialization.ByteArraySerializer;
import java.time.Duration;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The contents of a {@link TextMessage} containing data to give to a IOX Add-On over an RS232. It holds more data than
 * SerialIoxContent and is not compatible with all Add-Ons. For more information regarding Add-On compatible please
 * contact your reseller.
 *
 * <p>MimeContent is converted into bytes with a specific format. The first byte is the length of
 * the MimeType (N). The next N bytes are the ASCII encoded bytes of the MimeType string. The next two bytes are the
 * length of the Data (L). Finally, the next L bytes are the Data. Messages from MyGeotab will be delivered in this
 * format and messages to MyGeotab must be in this format as well.
 */
@Data
@NoArgsConstructor
public class MimeContent extends TextMessageContentType {

  /**
   * Gets or sets the media type of content contained in the data field. Free string, Maximum 255 characters.
   */
  private String mimeType;

  /**
   * The raw bytes to either send to an Add-On or that were received from an Add-On. Maximum 32767 bytes can be sent in
   * a single message.
   */
  @JsonDeserialize(using = ByteArrayDeserializer.class)
  @JsonSerialize(using = ByteArraySerializer.class)
  private Byte[] data;

  /**
   * The channel number to send to an Add-On or that were received from an Add-On.
   */
  private Integer channelNumber;

  /**
   * The delay to use between sending binary data packets to the GO Device.
   */
  @Deprecated
  private Duration binaryDataPacketDelay;

  /**
   * The component type.
   */
  private Integer componentType;

  @Builder(builderMethodName = "mimeContentBuilder")
  public MimeContent(String mimeType, Byte[] data, Integer channelNumber, Integer componentType) {
    super(MessageContentType.MIME_CONTENT);
    this.mimeType = mimeType;
    this.data = data;
    this.channelNumber = channelNumber;
    this.componentType = componentType;
  }
}
