/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.trip;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.geotab.model.coordinate.Coordinate;
import com.geotab.model.entity.Entity;
import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.user.Driver;
import com.geotab.model.entity.user.User;
import com.geotab.model.serialization.DriverEmbeddedSerializer;
import java.time.Duration;
import java.time.LocalDateTime;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * This is a vehicles trip and a summary of the driving activity during that trip.
 *
 * <p>To get more information about stops during a trip please see ExceptionEvent.
 *
 * <p>A complete "trip" is defined as starting at the time in which the vehicle starts and begins
 * being driven. The trip continues after the vehicle has been stopped and ends at the time the vehicle restarts and
 * begins being driven again; which then starts the next trip.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Trip extends Entity {

  /**
   * The duration between the start and stop of the trip.
   */
  private Duration drivingDuration;

  /**
   * The date and time that the trip started.
   */
  private LocalDateTime start;

  /**
   * The date and time that the trip stopped.
   */
  private LocalDateTime stop;

  /**
   * The distance the vehicle was driven after work hours (in km).
   */
  private Float afterHoursDistance;

  /**
   * The duration the vehicle was driven after work hours.
   */
  private Duration afterHoursDrivingDuration;

  /**
   * Whether the trip ends after hours.
   */
  private boolean afterHoursEnd;

  /**
   * Whether the trip starts after hours.
   */
  private boolean afterHoursStart;

  /**
   * The duration the vehicle was stopped after work hours.
   */
  private Duration afterHoursStopDuration;

  /**
   * Average speed in km/h. This only includes the average speed while driving.
   */
  private Float averageSpeed;

  /**
   * The distance the vehicle was driven during this trip (in km).
   */
  private Float distance;

  /**
   * Total end of trip idling (idling is defined as speed is 0 and ignition on). It is calculated from the beginning of
   * this trip to beginning of next trip.
   */
  private Duration idlingDuration;

  /**
   * Does this trip have a Seat Belt Off Incident.
   */
  @JsonProperty("isSeatBeltOff")
  private Boolean isSeatBeltOff;

  /**
   * The maximum speed of the vehicle during this trip (in km/h).
   */
  private Float maximumSpeed;

  /**
   * The start date of the next trip.
   */
  private LocalDateTime nextTripStart;

  /**
   * The number of incidents where the vehicle reached the first range of speeding triggers.
   */
  private Integer speedRange1;

  /**
   * The duration where the vehicle drove in the first range of speeding triggers.
   */
  private Duration speedRange1Duration;

  /**
   * The number of incidents where the vehicle reached the second range of speeding triggers.
   */
  private Integer speedRange2;

  /**
   * The duration where the vehicle drove in the second range of speeding triggers.
   */
  private Duration speedRange2Duration;

  /**
   * The number of incidents where the vehicle reached the third range of speeding triggers.
   */
  private Integer speedRange3;

  /**
   * The duration where the vehicle drove in the third range of speeding triggers.
   */
  private Duration speedRange3Duration;

  /**
   * The duration the vehicle was stopped at the end of the trip. This also includes any idling done at the end of a
   * trip.
   */
  private Duration stopDuration;

  /**
   * The {@link Coordinate} associated with this Trip's stop.
   */
  private Coordinate stopPoint;

  /**
   * The distance the vehicle was driven during work hours.
   */
  private Float workDistance;

  /**
   * The duration the vehicle was driven during work hours.
   */
  private Duration workDrivingDuration;

  /**
   * The duration the vehicle was stopped during work hours.
   */
  private Duration workStopDuration;

  /**
   * The {@link Device} associated with the trip.
   */
  private Device device;

  /**
   * The {@link Driver} for the trip.
   */
  @JsonSerialize(using = DriverEmbeddedSerializer.class)
  private User driver;
}
