/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.entity.trip;

import com.fasterxml.jackson.annotation.JsonEnumDefaultValue;
import com.fasterxml.jackson.annotation.JsonValue;
import com.geotab.model.serialization.HasName;
import com.geotab.util.Util;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public enum TripType implements HasName {

  /**
   * The device does not have a specific TripType assigned.
   */
  @JsonEnumDefaultValue
  UNKNOWN("Unknown", 0),

  /**
   * The device is in private mode and will have it's GPS data hidden.
   */
  PRIVATE("Private", 1),

  /**
   * The device is a special NoGPS device. This device will report speed but not position.
   */
  NO_GPS_DEVICE("NoGpsDevice", 2),

  /**
   * While a device has this TripType it will not send any positional or speed data.
   */
  NO_GPS_OR_SPEED_DEVICE("NoGpsOrSpeedDevice", 3);

  private final String name;
  private final int code;

  TripType(String name, int code) {
    this.name = name;
    this.code = code;
  }

  public int getCode() {
    return code;
  }

  @JsonValue
  public String getName() {
    return name;
  }

  public static TripType findOrDefault(String name) {
    if (Util.isEmpty(name)) {
      log.warn("Empty value is not recognized for {} enum; returning default {}",
          TripType.class.getSimpleName(), TripType.UNKNOWN);
      return UNKNOWN;
    }

    for (TripType tripType : values()) {
      if (tripType.getName().equalsIgnoreCase(name.trim())) {
        return tripType;
      }
    }

    log.warn("{} is not recognized for {} enum; returning default {}",
        name, TripType.class.getSimpleName(), TripType.UNKNOWN);
    return UNKNOWN;
  }
}
