/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.login;

import com.geotab.http.request.param.Parameters;
import com.geotab.util.Util;
import com.geotab.util.UtilityValidator;
import java.util.Objects;
import java.util.Optional;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

/**
 * The authentication credentials for a user when making calls to MyGeotab.
 */
@Data
@NoArgsConstructor
@SuperBuilder
public class Credentials extends Parameters {

  private String userName;
  private String password;
  private String database = "";
  private String sessionId;

  public Credentials(String userName, String password, String database, String sessionId) {
    if (password == null && sessionId == null) {
      throw new IllegalArgumentException("Must provide either password or session ID");
    }
    this.database = UtilityValidator.createDatabaseNameFromCompany(Optional.ofNullable(database).orElse(""));
    this.userName = userName;
    this.password = password;
    this.sessionId = sessionId;
  }

  public void setPassword(String password) {
    this.password = password;
    this.sessionId = null;
  }

  public void validate() {
    if (Util.isEmpty(userName)) {
      throw new IllegalArgumentException("Invalid user name");
    }

    if (Util.isEmpty(password) && Util.isEmpty(sessionId)) {
      throw new IllegalArgumentException("Must supply either password or sessionId");
    }
  }

  @Override
  public boolean equals(Object other) {
    if (this == other) {
      return true;
    }
    if (other == null || getClass() != other.getClass()) {
      return false;
    }
    Credentials that = (Credentials) other;
    return Objects.equals(userName, that.userName)
        && Objects.equals(database, that.database)
        && Objects.equals(sessionId, that.sessionId);
  }

  @Override
  public int hashCode() {
    return Objects.hash(userName, database, sessionId);
  }

  @Override
  public String toString() {
    return userName + "@" + database;
  }
}
