/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.model.entity.dvirlog.DVIRLog;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link DVIRLog}(s). A trailerSearch and deviceSearch
 * cannot be used at the same time because a DVIR log entry is only ever associated with one asset type (for instance,
 * if the "device" is set, "trailer" is always null and vice versa).
 */
@Data
@NoArgsConstructor
@SuppressWarnings("AbbreviationAsWordInName")
public class DVIRLogSearch implements Search {

  /**
   * Search for {@link DVIRLog} with this UserSearch Id
   *
   * <p>Available UserSearch options are:.
   * <ul>
   *    <li>Id</li>
   * </ul>
   */
  private UserSearch certifiedBySearch;

  /**
   * Gets or sets search for DVIRLogs that are a member of these defect Groups.
   *
   * <p>Available Group options are:.
   * <ul>
   *    <li>Id</li>
   * </ul>
   */
  private List<GroupSearch> defectSearch;

  /**
   * Filter by the DeviceSearch options. Providing a device ID will search for any DVIRLogs that are assigned to that
   * Device. Providing the Groups will search for DVIRLogs for that have Devices in that group.
   *
   * <p>Available DeviceSearch options are:.
   * <ul>
   *    <li>Id</li>
   *    <li>Groups</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * Search for DVIRLogs that were recorded at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Search for DVIRLogs that were recorded at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Search for DVIRLogs which are safe or are not safe to operate.
   */
  @JsonProperty("isSafeToOperate")
  private Boolean isSafeToOperate;

  /**
   * Search for {@link DVIRLog} repaired by a User.
   *
   * <p>Available UserSearch options are:.
   * <ul>
   *    <li>Id</li>
   * </ul>
   */
  private UserSearch repairedBySearch;

  /**
   * Filter by the TrailerSearch options. Providing a trailer ID will search for any DVIRLogs that are assigned to that
   * Trailer. Providing the Groups will search for DVIRLogs for that have Trailer in that group
   *
   * <p>Available TrailerSearch options are:
   * <ul>
   *     <li>
   *            Id
   *     </li>
   *     <li>
   *            Groups
   *     </li>
   * </ul>
   */
  private TrailerSearch trailerSearch;

  /**
   * Search for {@link DVIRLog} with this User.
   *
   * <p>Available UserSearch options are:.
   * <ul>
   *    <li>Id</li>
   * </ul>
   */
  private UserSearch userSearch;

  /**
   * Search for DVIRLogs and include boundary logs outside the from and to dates (for example, the log immediately
   * preceding the from date).
   */
  private Boolean includeBoundaryLogs;

  @Builder
  public DVIRLogSearch(UserSearch certifiedBySearch,
      List<GroupSearch> defectSearch, DeviceSearch deviceSearch, LocalDateTime fromDate,
      LocalDateTime toDate, Boolean isSafeToOperate,
      UserSearch repairedBySearch, TrailerSearch trailerSearch,
      UserSearch userSearch, Boolean includeBoundaryLogs) {
    this.certifiedBySearch = certifiedBySearch;
    this.defectSearch = defectSearch;
    this.deviceSearch = deviceSearch;
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.isSafeToOperate = isSafeToOperate;
    this.repairedBySearch = repairedBySearch;
    this.trailerSearch = trailerSearch;
    this.userSearch = userSearch;
    this.includeBoundaryLogs = includeBoundaryLogs;
  }
}
