/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.customproperty.Property;
import com.geotab.model.entity.customproperty.PropertySet;
import com.geotab.model.entity.customproperty.PropertyValue;
import com.geotab.model.entity.device.DeviceType;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

@Data
@NoArgsConstructor
public class DeviceSearch extends IdSearch {

  /**
   * Gets or sets search for Devices with comments matching this value. Wildcard can be used by prepending/appending "%"
   * to String. Example "%comments%".
   */
  private String comment;

  /**
   * Gets or sets search for Devices of this {@link DeviceType}.
   */
  private DeviceType deviceType;

  /**
   * Gets or sets search for Devices that were active at this date or after. Set to UTC now to search for only currently
   * active (non-archived) devices.
   */
  private LocalDateTime fromDate;

  /**
   * Gets or sets search for Devices that are a member of these {@link GroupSearch}(s). Each GroupSearch is an object
   * within the array.
   *
   * <p>Available GroupSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private List<GroupSearch> groups;

  /**
   * Gets or sets search for a Device with this unique hardware ID.
   */
  private Long hardwareId;

  /**
   * Gets or sets search for Devices with a license plate matching this value. Wildcard can be used by
   * prepending/appending "%" to String. Example "%comments%".
   */
  private String licensePlate;

  /**
   * Gets or sets search for Devices with this Name. Name is the primary description of the Device. Wildcard can be used
   * by prepending/appending "%" to String. Example "%name%".
   */
  private String name;

  /**
   * Gets or sets search for a Device by its unique serial number.
   */
  private String serialNumber;

  /**
   * Gets or sets search for Devices that were active at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Gets or sets search for a Device by Vehicle Identification Number (VIN). This is the unique number assigned to the
   * vehicle during manufacturing.
   */
  private String vehicleIdentificationNumber;

  /**
   * Gets or sets search for entities that contain specific keywords in all wildcard String-searchable fields.
   */
  private List<String> keywords;

  /**
   * Search for Devices that contain certain {@link Property} and/or {@link PropertyValue} and/or belong to certain
   * {@link PropertySet}.
   */
  private CustomPropertySearch customPropertySearch;

  @Builder
  @SuppressWarnings("squid:S00107")
  public DeviceSearch(String id, String comment, DeviceType deviceType,
      LocalDateTime fromDate, List<GroupSearch> groups, Long hardwareId, String licensePlate,
      String name, String serialNumber, LocalDateTime toDate, String vehicleIdentificationNumber,
      List<String> keywords, CustomPropertySearch customPropertySearch) {
    super(id);
    this.comment = comment;
    this.deviceType = deviceType;
    this.fromDate = fromDate;
    this.groups = groups;
    this.hardwareId = hardwareId;
    this.licensePlate = licensePlate;
    this.name = name;
    this.serialNumber = serialNumber;
    this.toDate = toDate;
    this.vehicleIdentificationNumber = vehicleIdentificationNumber;
    this.keywords = keywords;
    this.customPropertySearch = customPropertySearch;
  }
}
