/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.driverchange.DriverChange;
import com.geotab.model.entity.driverchange.DriverChangeType;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link DriverChange}.
 *
 * <p>When searching for status data including DeviceSearch and DiagnosticSearch the system will
 * return all records that match the search criteria and interpolate the value at the provided from/to dates when there
 * is no record that corresponds to the date. Interpolated records are dynamically created when the request is made and
 * can be identified as not having the ID property populated. Records with an ID are stored in the database.
 *
 * <p>This search has been designed to work efficiently with these combinations of parameters:
 * <ul>
 *      <li>
 *             Id
 *     </li>
 *      <li>
 *             DeviceSearch + DiagnosticSearch + FromDate and/or ToDate
 *     </li>
 * </ul>
 *
 * <p>Cannot be used with DriverGroups.
 */
@Data
@NoArgsConstructor
public class DriverChangeSearch extends IdSearch {

  /**
   * Search for {@link DriverChange} recorded for this DeviceSearch Id or in the provided Groups
   *
   * <p>Available DeviceSearch options are:.
   * <ul>
   *    <li>Id</li>
   *    <li>Group</li>
   *    <li>GroupListSearch</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * Search for {@link DriverChange} with this UserSearch Id or in the provided Groups
   *
   * <p>Available UserSearch options are:.
   * <ul>
   *    <li>Id</li>
   *    <li>DriverGroups</li>
   *    <li>DriverGroupListSearch</li>
   * </ul>
   */
  private UserSearch userSearch;

  /**
   * Search for {@link DriverChange} records that were logged at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Search for {@link DriverChange} records that were logged at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * The value indicating whether to include the last driver change before the from date or the most recent driver
   * change (if dates are not provided).
   */
  private boolean includeOverlappedChanges;

  /**
   * The value indicating the {@link DriverChangeType} to search for exclusively.
   */
  private DriverChangeType type;

  @Builder
  public DriverChangeSearch(String id, DeviceSearch deviceSearch,
      UserSearch userSearch, LocalDateTime fromDate, LocalDateTime toDate,
      boolean includeOverlappedChanges, DriverChangeType type) {
    super(id);
    this.deviceSearch = deviceSearch;
    this.userSearch = userSearch;
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.includeOverlappedChanges = includeOverlappedChanges;
    this.type = type;
  }
}
