/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.device.Device;
import com.geotab.model.entity.group.Group;
import com.geotab.model.entity.route.Route;
import com.geotab.model.entity.route.RouteType;
import com.geotab.model.entity.zone.Zone;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link Route}(s).
 */
@Data
@NoArgsConstructor
public class RouteSearch extends IdSearch {

  /**
   * Search for Routes that were active at this date or after.
   */
  private LocalDateTime fromDate;

  /**
   * Search for Routes that were active at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Search for Routes with this Name. Wildcard can be used by prepending/appending "%" to string. Example
   * "%comments%".
   */
  private String name;

  /**
   * Search for Routes with this {@link RouteType}.
   */
  private RouteType routeType;

  /**
   * Filter by the {@link ZoneSearch} options. Providing a zone ID will search for any Routes that contain that {@link
   * Zone}. Providing the Groups will search for Routes for that have {@link Zone}s in that {@link Group}.
   *
   * <p>Available {@link ZoneSearch} options are:
   * <ul>
   *    <li>id</li>
   * </ul>
   * <ul>
   *    <li>groups</li>
   * </ul>
   */
  private ZoneSearch zoneSearch;

  /**
   * Filter by the {@link DeviceSearch} options. Providing a device ID will search for any Routes that are assigned to
   * that {@link Device}. Providing the Groups will search for Routes for that have {@link Device}s in that {@link
   * Group}.
   *
   * <p>Available {@link DeviceSearch} options are:
   * <ul>
   *    <li>id</li>
   * </ul>
   * <ul>
   *    <li>groups</li>
   * </ul>
   *
   * <p>Only {@link RouteType#PLAN} have a {@link Device} assigned.
   */
  private DeviceSearch deviceSearch;

  @Builder
  public RouteSearch(String id, LocalDateTime fromDate, LocalDateTime toDate, String name,
      RouteType routeType, ZoneSearch zoneSearch, DeviceSearch deviceSearch) {
    super(id);
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.name = name;
    this.routeType = routeType;
    this.zoneSearch = zoneSearch;
    this.deviceSearch = deviceSearch;
  }
}
