package com.geotab.model.search;

import com.geotab.model.Id;
import com.geotab.model.entity.tachograph.files.TachographDataFile;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link TachographDataFile}.
 */
@Data
@NoArgsConstructor
public class TachographDataFileSearch extends IdSearch {

  /**
   * Gets or sets filter by the {@link DeviceSearch} options.
   * Providing a device ID will search for any file that is assigned to that Device.
   * Providing the Groups will search for files that have Devices in that group.
   * Providing the device IDs will search for files that have Devices in that list.
   * Available DeviceSearch options are:
   * <list>
   * <item>
   * <description>Id</description>
   * </item>
   * <item>
   * <description>Group</description>
   * </item>
   * <item>
   * <description>DeviceIds</description>
   * </item>
   * </list>
   */
  private DeviceSearch deviceSearch;

  /**
   * Gets or sets filter by the {@link UserSearch} options.
   * Providing a user ID will search for any file that is assigned to that user.
   * Providing the Groups will search for files that have Users in that group.
   * Providing the user IDs will search for files that have Users in that list, in this case the users will be drivers.
   * Available UserSearch options are:
   * <list>
   * <item>
   * <description>Id</description>
   * </item>
   * <item>
   * <description>Group</description>
   * </item>
   * <item>
   * <description>UserIds</description>
   * </item>
   * </list>
   */
  private UserSearch userSearch;

  /**
   * Gets or sets search for file records that were uploaded at this date or after.
   */
  private LocalDateTime fromUploadDate;

  /**
   * Gets or sets search for file records that were uploaded at this date or before.
   */
  private LocalDateTime toUploadDate;

  /**
   * Gets or sets search for file records based on the file type.
   * The available values are:
   * <list>
   * <item>
   * <description>"Driver": To filter by file type Driver.</description>
   * </item>
   * <item>
   * <description>"Device": To filter by file type Device.</description>
   * </item>
   * </list>
   */
  private String type;

  /**
   * Gets or sets the flag to include binary data in the response.
   */
  private Boolean includeBinaryData;

  /**
   * Gets or sets search for TachographFile records that were generated at this date or after.
   */
  private LocalDateTime fromTimestamp;

  /**
   * Gets or sets search for TachographFile records that were generated at this date or before.
   */
  private LocalDateTime toTimestamp;

  /**
   * Gets or sets search for entities that contain specific keywords into FileName, FileNameDdd, FileNameTgd, FileNameV1B,
   * in case the “type = Driver” it will look into the driver's first name and last name, or when “type = Device” in the license plate.
   * Note: It is currently limited to only one keyword. To use keywords it is necessary to use the “type” filter too.
   */
  private List<String> keywords;

  /**
   * Gets or sets the row version of the
   * {@link TachographDataFile}
   * search criteria.
   */
  private long version;

  /** Gets or sets search for TachographFile Device Ids records are in the list. */
  private Id[] ids;

  /** Gets or sets search for files with a specific processing status. */
  private int processingStatus;

  /**
   * Gets or sets the filter to search the files in a period of time from floor to ceiling.
   * Available FloorToCeilingTimestampSearch options are:
   * <list>
   * <item>
   * <description>FromTimestamp: Gets or sets search for TachographFile records that were generated at this date or after.</description>
   * </item>
   * <item>
   * <description>ToTimestamp: Gets or sets search for TachographFile records that were generated at this date or before.</description>
   * </item>
   * </list>
   */
  private FloorToCeilingTimestampSearch fromFloorToCeilingTimestamp;

  /**
   * Gets or sets search for files without errors.
   */
  private Boolean withoutError;

  /**
   * Gets or sets search for contains text in errors.
   */
  private String errorContains;

  /**
   * Gets or sets the flag to include archived files. By default archived tachograph data file records are not returned, set this flag to true to
   * return all records.
   */
  private Boolean includeArchived;

  /**
   * Gets or sets LocalDateTime to filter by ActivityTimestampEnd.
   * It will return the last file of each driver where ActivityTimestampEnd is greater than this value.
   */
  private LocalDateTime fromActivityTimestampEnd;

  /**
   * Gets or sets the filter to search the files in a period of activity time.
   * Available ActivityTimestampBetweenSearch options are:
   * <list>
   * <item>
   * <description>FromActivityTimestamp: Gets or sets search for TachographFile records whose activity period starts.</description>
   * </item>
   * <item>
   * <description>ToActivityTimestamp: Gets or sets search for TachographFile records whose activity period ends.</description>
   * </item>
   * </list>
   */
  private ActivityTimestampBetweenSearch activityTimestampBetween;

  @Builder
  public TachographDataFileSearch(String id, DeviceSearch deviceSearch, UserSearch userSearch, LocalDateTime fromUploadDate,
      LocalDateTime toUploadDate, String type, Boolean includeBinaryData, LocalDateTime fromTimestamp, LocalDateTime toTimestamp,
      List<String> keywords, long version, Id[] ids, int processingStatus, FloorToCeilingTimestampSearch fromFloorToCeilingTimestamp,
      Boolean withoutError, String errorContains, Boolean includeArchived, LocalDateTime fromActivityTimestampEnd,
      ActivityTimestampBetweenSearch activityTimestampBetween) {
    super(id);
    this.deviceSearch = deviceSearch;
    this.userSearch = userSearch;
    this.fromUploadDate = fromUploadDate;
    this.toUploadDate = toUploadDate;
    this.type = type;
    this.includeBinaryData = includeBinaryData;
    this.fromTimestamp = fromTimestamp;
    this.toTimestamp = toTimestamp;
    this.keywords = keywords;
    this.version = version;
    this.ids = ids;
    this.processingStatus = processingStatus;
    this.fromFloorToCeilingTimestamp = fromFloorToCeilingTimestamp;
    this.withoutError = withoutError;
    this.errorContains = errorContains;
    this.includeArchived = includeArchived;
    this.fromActivityTimestampEnd = fromActivityTimestampEnd;
    this.activityTimestampBetween = activityTimestampBetween;
  }

  /**
   * Filter to search the files in a period of activity time.
   */
  @Data
  @NoArgsConstructor
  public static class ActivityTimestampBetweenSearch {
    /**
     * Gets or sets search for TachographFile records whose activity period starts.
     */
    private LocalDateTime fromActivityTimestamp;

    /**
     * Gets or sets search for TachographFile records whose activity period ends.
     */
    private LocalDateTime toActivityTimestamp;
  }

  /**
   * Filter to search the files in a period of time from floor to ceiling.
   */
  @Data
  @NoArgsConstructor
  public static class FloorToCeilingTimestampSearch {
    /**
     * Gets or sets search for TachographFile records that were generated at this date or after.
     */
    private LocalDateTime fromTimestamp;

    /**
     * Gets or sets search for TachographFile records that were generated at this date or before.
     */
    private LocalDateTime toTimestamp;
  }
}
