/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.trip.Trip;
import java.time.LocalDateTime;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link Trip}(s).
 *
 * <p>This search has been designed to work efficiently with these combinations of parameters:
 * <ul>
 *      <li>
 *             Id
 *     </li>
 *      <li>
 *             DeviceSearch + FromDate and/or ToDate (+ IncludeOverlappedTrips)
 *     </li>
 *      <li>
 *             UserSearch + FromDate and/or ToDate (+ IncludeOverlappedTrips)
 *     </li>
 * </ul>
 *
 * <p>Cannot be used with DriverGroups.
 */
@Data
@NoArgsConstructor
public class TripSearch extends IdSearch {

  /**
   * Search for {@link Trip} with this DeviceSearch Id
   *
   * <p>Available DeviceSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private DeviceSearch deviceSearch;

  /**
   * Search for {@link Trip} with this UserSearch Id
   *
   * <p>Available UserSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private UserSearch userSearch;

  /**
   * Search for Trips recorded at this date or after. When "IncludeOverlappedTrips" is set to True, search for Trips
   * where the NextTripStartTime is at this date, after or NULL.
   */
  private LocalDateTime fromDate;

  /**
   * Search for Trips recorded at this date or before. When "IncludeOverlappedTrips" is set to True, search for Trips
   * where the StartDateTime is this date or before.
   */
  private LocalDateTime toDate;

  /**
   * A value indicating whether when OverlappedTrips is set to True; any part of a trip that overlaps with the FromDate
   * or ToDate boundary will have the entire trip included in the data.
   */
  private Boolean includeOverlappedTrips;

  @Builder
  public TripSearch(String id, DeviceSearch deviceSearch,
      UserSearch userSearch, LocalDateTime fromDate, LocalDateTime toDate,
      Boolean includeOverlappedTrips) {
    super(id);
    this.deviceSearch = deviceSearch;
    this.userSearch = userSearch;
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.includeOverlappedTrips = includeOverlappedTrips;
  }
}
