/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.geotab.model.entity.user.HosRuleSet;
import com.geotab.model.entity.user.UserAuthenticationType;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for a {@link com.geotab.model.entity.user.User} or {@link
 * com.geotab.model.entity.user.Driver}.
 */
@Data
@NoArgsConstructor
public class UserSearch extends IdSearch {

  /**
   * Search for Users who are a member of this {@link GroupSearch}.
   *
   * <p>Available GroupSearch options are:
   * <ul>
   *      <li>
   *             GroupSearch.id
   *     </li>
   * </ul>
   *
   * <p>Cannot be used with DriverGroups.
   */
  private List<GroupSearch> companyGroups;

  /**
   * Search for Users who are assigned a Driver Key which is a member of the {@link GroupSearch}.
   *
   * <p>Available GroupSearch options are:
   * <ul>
   *      <li>
   *             GroupSearch.id
   *     </li>
   * </ul>
   *
   * <p>Cannot be used with CompanyGroups.
   */
  private List<GroupSearch> driverGroups;

  /**
   * Search for Users with this first name. Wildcard can be used by prepending/appending "%" to string. Example
   * "%firstName%". This property is negatable. If the first character of this search property is '!', then the API will
   * know to negate the search logic. (e.g. field = "!John%", is equivalent to: WHERE NOT LIKE 'John%')
   */
  private String firstName;

  /**
   * Gets or sets search for Users that were active at this date or after. Set to UTC now to search for only currently
   * active (non-archived) users.
   */
  private LocalDateTime fromDate;

  /**
   * Only search for Users who have a Driver Key assigned.
   */
  @JsonProperty("isDriver")
  private Boolean isDriver;

  /**
   * Search for a User who is associated with this Driver Key Id.
   */
  private Long keyId;

  /**
   * Search for Users with this last name. Wildcard can be used by prepending/appending "%" to string. Example
   * "%lastName%". This property is negatable. If the first character of this search property is '!', then the API will
   * know to negate the search logic. (e.g. field = "!John%", is equivalent to: WHERE NOT LIKE 'John%')
   */
  private String lastName;

  /**
   * search for Users with this email/log-on name. Wildcard can be used by prepending/appending "%" to string. Example
   * "%name%". This property is negatable. If the first character of this search property is '!', then the API will know
   * to negate the search logic. (e.g. field = "!John%", is equivalent to: WHERE NOT LIKE 'John%')
   */
  private String name;

  /**
   * Search for a User who is associated with this Driver Serial Number.
   */
  private String serialNumber;

  /**
   * Search for Users that were active at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Search for entities that contain specific keywords in all wildcard string-searchable fields.
   */
  private List<String> keywords;

  /**
   * Search for Users who are assigned to a specific Security Clearance which is a member of the {@link GroupSearch}.
   *
   * <p>Available GroupSearch options are:
   * <ul>
   *      <li>
   *             GroupSearch.id
   *     </li>
   * </ul>
   *
   * <p>Cannot be used with CompanyGroups.
   */
  private List<GroupSearch> securityGroups;

  /**
   * Search for a User who is associated with this Driver License Number. Wildcard can be used by prepending/appending
   * "%" to string. Example "%LicenseNumber%". This property is negatable. If the first character of this search
   * property is '!', then the API will know to negate the search logic. (e.g. field = "!LicenseNumber%", is equivalent
   * to: WHERE NOT LIKE 'LicenseNumber%')
   */
  private String licenseNumber;

  /**
   * Search for a User who is associated with this Driver Employee Number. Wildcard can be used by prepending/appending
   * "%" to string. Example "%EmployeeNumber%". This property is negatable. If the first character of this search
   * property is '!', then the API will know to negate the search logic. (e.g. field = "!EmployeeNumber%", is equivalent
   * to: WHERE NOT LIKE 'EmployeeNumber%')
   */
  private String employeeNumber;

  /**
   * Search for Users who are associated with this {@link HosRuleSet}.
   */
  private HosRuleSet hosRuleSet;

  /**
   * Search for Users who are associated with this {@link UserAuthenticationType}.
   */
  private UserAuthenticationType authenticationType;

  @Builder
  public UserSearch(String id, List<GroupSearch> companyGroups,
      List<GroupSearch> driverGroups, String firstName, LocalDateTime fromDate, Boolean isDriver,
      Long keyId, String lastName, String name, String serialNumber, LocalDateTime toDate,
      List<String> keywords, List<GroupSearch> securityGroups, String licenseNumber,
      String employeeNumber, HosRuleSet hosRuleSet,
      UserAuthenticationType authenticationType) {
    super(id);
    this.companyGroups = companyGroups;
    this.driverGroups = driverGroups;
    this.firstName = firstName;
    this.fromDate = fromDate;
    this.isDriver = isDriver;
    this.keyId = keyId;
    this.lastName = lastName;
    this.name = name;
    this.serialNumber = serialNumber;
    this.toDate = toDate;
    this.keywords = keywords;
    this.securityGroups = securityGroups;
    this.licenseNumber = licenseNumber;
    this.employeeNumber = employeeNumber;
    this.hosRuleSet = hosRuleSet;
    this.authenticationType = authenticationType;
  }
}
