/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.search;

import com.geotab.model.entity.zone.IncludeGroups;
import com.geotab.model.entity.zone.Zone;
import com.geotab.model.geographical.BoundingBox;
import java.time.LocalDateTime;
import java.util.List;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

/**
 * The object used to specify the arguments when searching for {@link Zone}(s).
 */
@Data
@NoArgsConstructor
public class ZoneSearch extends IdSearch {

  /**
   * Search for Zones with this External Reference. Wildcard can be used by prepending/appending "%" to string. Example
   * "%reference%".
   */
  private String externalReference;

  /**
   * Search for Zones that are members of these {@link GroupSearch}(s) one of it's children or one of it's parents.
   *
   * <p>Available GroupSearch options are:.
   * <ul>
   *    <li>id</li>
   * </ul>
   */
  private List<GroupSearch> groups;

  /**
   * Include zones that are in the in this hierarchy of the {@link GroupSearch}(s) provided.
   *
   * <p>If no {@link GroupSearch}(s) are provided the user's data scope groups will be used.
   *
   * <p>Default: {@link IncludeGroups}.ParentAndChild.
   */
  private IncludeGroups includeGroups;

  /**
   * Exclude Zones whose radius is smaller than this size (meters).
   */
  private Integer minimumRadiusInMeters;

  /**
   * Search for Zones with this Name. Wildcard can be used by prepending/appending "%" to string. Example "%name%".
   */
  private String name;

  /**
   * The {@link BoundingBox} search for Zones in this area extent, the zones being retrieved must be located in this
   * area. Typically used for retrieving Zones in the extents of a bounding box. The SearchArea object should contain
   * the minimum and maximum latitude and longitude representing the search area.
   */
  private BoundingBox searchArea;

  /**
   * Search for Zones that were active at this date or after. Set to UTC now to search for only currently active
   * (non-archived) zones.
   */
  private LocalDateTime fromDate;

  /**
   * Search for Zones that were active at this date or before.
   */
  private LocalDateTime toDate;

  /**
   * Search for entities that contain specific keywords in all wildcard string-searchable fields.
   */
  private List<String> keywords;

  @Builder
  public ZoneSearch(String id, String externalReference,
      List<GroupSearch> groups, IncludeGroups includeGroups, Integer minimumRadiusInMeters,
      String name, BoundingBox searchArea, LocalDateTime fromDate, LocalDateTime toDate,
      List<String> keywords) {
    super(id);
    this.externalReference = externalReference;
    this.groups = groups;
    this.includeGroups = includeGroups;
    this.minimumRadiusInMeters = minimumRadiusInMeters;
    this.name = name;
    this.searchArea = searchArea;
    this.fromDate = fromDate;
    this.toDate = toDate;
    this.keywords = keywords;
  }
}
