/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.model.serialization;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import java.io.IOException;
import lombok.extern.slf4j.Slf4j;

/**
 * Custom system entity serializer used by default for all classes implementing {@link SystemEntitySerializationAware}
 * and which should return true for {@link SystemEntitySerializationAware#isSystemEntity()}. A {@link
 * SystemEntitySerializationAware} is serialized to a string containing the entity id only.
 *
 * <p>If another custom serializer is defined for the class which implements {@link
 * SystemEntitySerializationAware}, then that one will take precedence. Make sure you serialize the system entity the
 * same way.
 *
 * <p>If another custom serializer is defined for the parent class, but the child class applies
 * {@link SystemEntitySerializationAware}, then {@link SystemEntitySerializer} takes precedence for the child class.
 */
@Slf4j
public class SystemEntitySerializer extends JsonSerializer<SystemEntitySerializationAware> {

  @Override
  public void serialize(SystemEntitySerializationAware o, JsonGenerator jg, SerializerProvider sp) throws IOException {
    if (o.isSystemEntity()) {
      jg.writeString(o.getId().getId());
    } else {
      log.error("Expecting to serialize a system entity type,"
          + " but it's flag is set to false (marked as not a system entity)!");
    }
  }
}
