/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.util;

import static com.geotab.util.Util.listOf;
import static java.util.Optional.ofNullable;

import com.geotab.model.entity.device.CustomVehicleDevice;
import com.geotab.model.entity.device.DigitalAuxiliariesAware;
import com.geotab.model.entity.device.EngineHoursAdjustmentProviderAware;
import com.geotab.model.entity.device.GoCurve;
import com.geotab.model.entity.device.GoDevice;
import com.geotab.model.entity.device.GoLegacy;
import com.geotab.model.entity.device.GoTalkLanguage;
import com.geotab.model.entity.device.LicensableAware;
import com.geotab.model.entity.device.OdometerAdjustmentProviderAware;
import com.geotab.model.entity.device.RfChannelsAware;
import com.geotab.model.entity.device.VehicleIdentificationNumberAware;
import com.geotab.model.entity.enginetype.EngineTypeGeneric;
import java.util.function.Consumer;
import lombok.experimental.UtilityClass;

/**
 * A helper class where all device defaults are contained. This allows us to reuse some of the defaults for non-related
 * device objects. For example, {@link GoLegacy} and Go4v3 both have auxiliaries but are not in the same lineage.
 */
@UtilityClass
public class DeviceDefaultsUtil {

  public static <T> T apply(T t, Consumer<T> fn) {
    fn.accept(t);
    return t;
  }

  /**
   * Add the defaults to VIN.
   */
  public static void addVehicleIdentificationNumberDefaults(VehicleIdentificationNumberAware device) {
    if (device == null) return;
    device.setVehicleIdentificationNumber(ofNullable(device.getVehicleIdentificationNumber()).orElse(""));
    device.setEngineVehicleIdentificationNumber(ofNullable(device.getEngineVehicleIdentificationNumber()).orElse("?"));
  }

  /**
   * Applies the defaults for the GoDevice device. We do not currently have an interface for the things the GoDevice
   * adds - but we need to centralize the settings so it's here as GoDevice for now.
   */
  public static void addGoDeviceDefaults(GoDevice device) {
    if (device == null) return;
    device.setEngineType(ofNullable(device.getEngineType()).orElse(EngineTypeGeneric.getInstance()));
    device.setMinor(ofNullable(device.getMinor()).orElse((short) 0));
    device.setMajor(ofNullable(device.getMajor()).orElse((short) 0));
    device.setAutoGroups(ofNullable(device.getAutoGroups()).orElse(listOf()));
    device.setMinAccidentSpeed(ofNullable(device.getMinAccidentSpeed()).orElse(4d));
    device.setSpeedingOn(ofNullable(device.getSpeedingOn()).orElse(100d));
    device.setSpeedingOff(ofNullable(device.getSpeedingOff()).orElse(90d));
    device.setGpsOffDelay(ofNullable(device.getGpsOffDelay()).orElse(0));
    device.setParameterVersion(ofNullable(device.getParameterVersion()).orElse(0));
    device.setParameterVersionOnDevice(ofNullable(device.getParameterVersionOnDevice()).orElse((short) 0));
    device.setIdleMinutes(ofNullable(device.getIdleMinutes()).orElse(3));
    device.setGoTalkLanguage(ofNullable(device.getGoTalkLanguage()).orElse(GoTalkLanguage.ENGLISH));
    device.setFuelTankCapacity(ofNullable(device.getFuelTankCapacity()).orElse(0d));
    device.setWifiHotspotLimits(ofNullable(device.getWifiHotspotLimits()).orElse(listOf()));
    device.setCustomParameters(ofNullable(device.getCustomParameters()).orElse(listOf()));
    addLicensableDefaults(device);
  }

  /**
   * Adds the license defaults.
   */
  public static void addLicensableDefaults(LicensableAware device) {
    if (device == null) return;
    device.setLicensePlate(ofNullable(device.getLicensePlate()).orElse(""));
    device.setLicenseState(ofNullable(device.getLicenseState()).orElse(""));
  }

  /**
   * Adds the go legacy defaults.
   */
  public static void addGoLegacyDefaults(GoLegacy device) {
    if (device == null) return;
    device.setDeltaSpeed(ofNullable(device.getDeltaSpeed()).orElse(9d));
    device.setDeltaMinSpeed(ofNullable(device.getDeltaMinSpeed()).orElse(50d));
    device.setDeltaHeading(ofNullable(device.getDeltaHeading()).orElse(48));
    device.setDeltaHeadingMinSpeed(ofNullable(device.getDeltaHeadingMinSpeed()).orElse(20d));
    device.setDeltaHeadingHS(ofNullable(device.getDeltaHeadingHS()).orElse(14));
    device.setDeltaHeadingMinSpeedHS(ofNullable(device.getDeltaHeadingMinSpeedHS()).orElse(72d));
    device.setDeltaDistance(ofNullable(device.getDeltaDistance()).orElse(1000));
    device.setHarshBreak(ofNullable(device.getHarshBreak()).orElse(17d));
  }

  /**
   * Adds the RF channel defaults.
   */
  public static void addRfChannelDefaults(RfChannelsAware device) {
    if (device == null) return;
    device.setChannelCount(ofNullable(device.getChannelCount()).orElse((byte) 1));
    device.setChannel(GoDevice.checkAndPadChannelArray(device.getChannel()));
    device.setFrequencyOffset(ofNullable(device.getFrequencyOffset()).orElse((byte) 1));
    device.setRfParameterVersion((short) 0);
  }

  /**
   * Adds the auxiliary defaults.
   */
  public static void addAuxiliaryDefaults(DigitalAuxiliariesAware device) {
    if (device == null) return;
    device.setEnableAuxWarning(GoDevice.checkAndPadAuxArray(device.getEnableAuxWarning()));
    device.setAuxWarningSpeed(GoDevice.checkAndPadAuxArray(device.getAuxWarningSpeed()));
    device.setIsAuxIgnTrigger(GoDevice.checkAndPadAuxIgnArray(device.getIsAuxIgnTrigger()));
    device.setExternalDeviceShutDownDelay(0);
    device.setImmobilizeArming(3);
    device.setIsAuxInverted(GoDevice.checkAndPadAuxIgnArray(device.getIsAuxInverted()));
  }

  /**
   * Applies the defaults for the GoCurve device. We do not currently have an interface for the things the GoCurve adds
   * - but we need to centralize the settings so it's here as GoCurve for now.
   */
  public static void addGoCurveDefaults(GoCurve device) {
    if (device == null) return;
    device.setEnableBeepOnRpm(false);
    device.setRpmValue(3500);
    device.setDriverSeatbeltWarningOn(false);
    device.setPassengerSeatbeltWarningOn(false);
    device.setReverseDetectOn(false);
    device.setIoxConnectionEnabled(true);
    device.setAccelerationWarningThreshold(22);
    device.setBrakingWarningThreshold(-34);
    device.setCorneringWarningThreshold(26);
    device.setAccelerometerThresholdWarningFactor(0);
    device.setSeatbeltWarningSpeed(10d);
    device.setEnableBeepOnDangerousDriving(false);
    device.setActiveTrackingEnabled(false);
    addOdometerAdjustmentProviderDefaults(device);
    addEngineHoursAdjustmentProviderDefaults(device);
  }

  /**
   * Adds the odometer adjustment provider defaults.
   */
  public static void addOdometerAdjustmentProviderDefaults(OdometerAdjustmentProviderAware device) {
    if (device == null) return;
    device.setOdometerOffset(0d);
    device.setOdometerFactor(1f);
  }

  /**
   * Adds the odometer adjustment provider defaults.
   */
  public static void addEngineHoursAdjustmentProviderDefaults(EngineHoursAdjustmentProviderAware device) {
    if (device == null) return;
    device.setEngineHourOffset(0);
  }


  /**
   * Adds the custom vehicle device defaults.
   */
  public static void addCustomVehicleDeviceDefaults(CustomVehicleDevice device) {
    if (device == null) return;
    addVehicleIdentificationNumberDefaults(device);
    addOdometerAdjustmentProviderDefaults(device);
    addEngineHoursAdjustmentProviderDefaults(device);
    addLicensableDefaults(device);
  }
}
