/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.util;

import static com.geotab.util.Util.listOf;
import static com.geotab.util.Util.startWithIc;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import lombok.experimental.UtilityClass;

/**
 * This class contains helper methods related to encoding and parsing device serial numbers and hardware IDs.
 */
@UtilityClass
public class ThirdPartyHelper {

  /**
   * OEM device types.
   */
  public static final Map<Integer, String> THIRD_PARTY_DEVICE_TYPES =
      Util.<Integer, String>mapBuilder()
          .put(10000, "C1")
          .put(10001, "C2")
          .put(10002, "C3")
          .put(10003, "C4")
          .put(10004, "C5")
          .put(10005, "C6")
          .put(10006, "C7")
          .put(10007, "C8")
          .put(10008, "C9")
          .put(10009, "CA")
          .put(10010, "CB")
          .put(10011, "CC")
          .put(10012, "CD")
          .put(10013, "CE")
          .put(10014, "CF")
          .put(10015, "CG")
          .put(10016, "CH")
          .put(10017, "CI")
          .put(10018, "CJ")
          .put(10019, "CK")
          .put(10020, "CL")
          .put(10021, "CM")
          .put(10022, "CN")
          .put(10023, "CO")
          .put(10024, "CP")
          .put(10025, "CQ")
          .put(10026, "CR")
          .put(10027, "CS")
          .put(10028, "CT")
          .put(10029, "CU")
          .put(10030, "CV")
          .put(10031, "CW")
          .put(10032, "CX")
          .put(10033, "CY")
          .put(10034, "CZ")
          .put(10035, "D1")
          .put(10036, "D2")
          .put(10037, "D3")
          .put(10038, "D4")
          .put(10039, "D5")
          .put(10040, "D6")
          .put(10041, "D7")
          .put(10042, "D8")
          .put(10043, "D9")
          .put(10044, "DA")
          .put(10045, "DB")
          .put(10046, "DC")
          .put(10047, "DD")
          .put(10048, "DE")
          .put(10049, "DF")
          .put(10050, "DG")
          .put(10051, "DH")
          .put(10052, "DI")
          .put(10053, "DJ")
          .put(10054, "DK")
          .put(10055, "DL")
          .put(10056, "DM")
          .put(10057, "DN")
          .put(10058, "DO")
          .put(10059, "DP")
          .put(10060, "DQ")
          .put(10061, "DR")
          .put(10062, "DS")
          .put(10063, "DT")
          .put(10064, "DU")
          .put(10065, "DV")
          .put(10066, "DW")
          .put(10067, "DX")
          .put(10068, "DY")
          .put(10069, "DZ")
          .put(10070, "E1")
          .put(10071, "E2")
          .put(10072, "E3")
          .put(10073, "E4")
          .put(10074, "E5")
          .put(10075, "E6")
          .put(10076, "E7")
          .put(10077, "E8")
          .put(10078, "E9")
          .put(10079, "EA")
          .put(10080, "EB")
          .put(10081, "EC")
          .put(10082, "ED")
          .put(10083, "EE")
          .put(10084, "EF")
          .put(10085, "EG")
          .put(10086, "EH")
          .put(10087, "EI")
          .put(10088, "EJ")
          .put(10089, "EK")
          .put(10090, "EL")
          .put(10091, "EM")
          .put(10092, "EN")
          .put(10093, "EO")
          .put(10094, "EP")
          .put(10095, "EQ")
          .put(10096, "ER")
          .put(10097, "ES")
          .put(10098, "ET")
          .put(10099, "EU")
          .put(10100, "EV")
          .put(10101, "EW")
          .put(10102, "EX")
          .put(10103, "EY")
          .put(10104, "EZ")
          .build();

  /**
   * Third-party device types.
   */
  public static final Set<Integer> OEM_DEVICE_TYPES = new HashSet<>(listOf(
      10023,
      10042,
      10055,
      10056,
      10057,
      10058,
      10059,
      10060,
      10061,
      10062,
      10063,
      10064,
      10065,
      10066,
      10067,
      10068,
      10069,
      10070,
      10071,
      10072,
      10073,
      10074,
      10075,
      10076,
      10077,
      10078,
      10079,
      10080));

  /**
   * Gets the third-party product ID from a serial number or device prefix.
   *
   * @param value The serial number or device prefix.
   * @return The corresponding product ID, or 0 if the value is not a valid third-party serial number or device prefix.
   */
  public static int getThirdPartyProductId(String value) {
    if (Util.isEmpty(value)) {
      return 0;
    }
    for (Map.Entry<Integer, String> deviceType : THIRD_PARTY_DEVICE_TYPES.entrySet()) {
      if (startWithIc(deviceType.getValue(), value)) {
        return deviceType.getKey();
      }
    }
    return 0;
  }

  /**
   * Determines whether [is third party vehicle device] [the specified product identifier].
   *
   * @param productId the product identifier.
   * @return true if [is third party vehicle device] [the specified product identifier]; otherwise, false.
   */
  public static boolean isThirdPartyVehicleDevice(int productId) {
    return isOemDevice(productId)
        || isThirdPartyDevice(productId)
        && (productId == 10017 || productId == 10026 || productId == 10029
        || productId == 10033 || productId == 10040 || productId == 10042 || productId == 10044
        || productId == 10085 || productId == 10086 || productId == 10089);
  }

  /**
   * Checks if a serial number or device prefix is a supported third-party device type.
   *
   * @param value The serial number or device prefix.
   * @return True if the serial number or device prefix is a supported third-party device type, false otherwise.
   */
  public static boolean isThirdPartyDevice(String value) {
    return getThirdPartyProductId(value) > 0;
  }

  /**
   * Checks if a product ID is a supported third-party device type.
   *
   * @param productId The product ID.
   * @return True if the product ID is a supported third-party device type, false otherwise.
   */
  public static boolean isThirdPartyDevice(int productId) {
    return THIRD_PARTY_DEVICE_TYPES.containsKey(productId);
  }

  /**
   * Checks if a product ID is a supported OEM device type.
   *
   * @param productId The product ID.
   * @return True if the product ID is a supported OEM device type, false otherwise.
   */
  public static boolean isOemDevice(int productId) {
    return OEM_DEVICE_TYPES.contains(productId);
  }
}
