/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.util;

import static java.util.stream.Collectors.toList;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.jetbrains.annotations.Nullable;

/**
 * Internal use only.
 */
public interface Util {

  static boolean isEmpty(@Nullable Collection<?> collection) {
    return collection == null || collection.isEmpty();
  }

  static boolean isEmpty(@Nullable String str) {
    return str == null || str.isEmpty();
  }

  static boolean isNotEmpty(@Nullable String str) {
    return !isEmpty(str);
  }

  static boolean startWithIc(String prefix, @Nullable String str) {
    if (isEmpty(prefix)) throw new IllegalArgumentException("prefix required");
    if (isEmpty(str)) return false;
    //noinspection StringEquality
    if (prefix == str) return true;
    return str.length() >= prefix.length() && prefix.regionMatches(true, 0, str, 0, prefix.length());
  }

  static List<?> intersect(@Nullable List<?> firstList, List<?> secondList) {
    if (firstList == null || secondList == null) return Collections.EMPTY_LIST;
    return firstList.stream().distinct().filter(secondList::contains).collect(toList());
  }

  @SafeVarargs
  static <T> List<T> listOf(T... ts) {
    return new ArrayList<>(Arrays.asList(ts));
  }

  static <T, V> MapBuilder<T, V> mapBuilder() {
    return new MapBuilder<>();
  }

  class MapBuilder<T, V> {

    private final HashMap<T, V> out = new HashMap<>();

    public MapBuilder<T, V> put(T key, V value) {
      out.put(key, value);
      return this;
    }

    public Map<T, V> build() {
      return out;
    }
  }

  @FunctionalInterface
  interface FailableFunction<T, R, E extends Throwable> {

    R apply(T t) throws E;
  }
}
