/*
 *
 * 2020 Copyright (C) Geotab Inc. All rights reserved.
 */

package com.geotab.util;

import static com.geotab.util.Util.startWithIc;

import lombok.experimental.UtilityClass;

@UtilityClass
public class UtilityValidator {

  /**
   * Value of 58 copied from WebMethods.GetValidatedDatabaseName for consistency. Postgres documentation says the limit
   * is 63. Later consolidate all checks in one place. For now, just see what we need to do to pass security scanner
   * validation.
   */
  public static final int MAX_DB_NAME_LENGTH = 58;

  /**
   * Creates the database name from supplied company. Honours certain rules about naming databases
   *
   * <p>*** almost the same as CreateSafeDatabaseName ***
   *
   * <p>Some of the complex authentication code relies on certain exceptions being thrown.
   *
   * <p>throwing when the input isnull or empty is the "wrong" exception.
   *
   * <p>This function is here for legacy support, as the  first changes are to support Veracode
   * security scanning.
   *
   * <p>Logic changes should be a different ticket.
   *
   * @param original String used to create the database name.  For example, it could be based on the name of the
   *     company.
   * @return Sanitized company name.
   */
  public static String createDatabaseNameFromCompany(String original) {

    // legacy: don't check for IsNullOrEmpty
    char[] origChars = original.toCharArray();
    int origLength = origChars.length;
    char[] newChars = new char[original.length()];
    int newLength = 0;

    for (int num = 0; num < origLength && newLength < MAX_DB_NAME_LENGTH; num++) {
      char c = origChars[num];

      // Only check for underscore, not period or colon
      if (Character.isLetterOrDigit(c) || c == '_') {
        newChars[newLength] = c;
        newLength++;
        // Invalid chars become underscores, but only one underscore in a row.
      } else if (newLength > 0 && newChars[newLength - 1] != '_') {
        newChars[newLength] = '_';
        newLength++;
      }
    }

    // Cannot finish with a trailing underscore.
    if (newLength > 0 && newChars[newLength - 1] == '_') {
      newLength--;
    }

    String databaseName = new String(newChars, 0, newLength);

    // The database name can't look like a postgres table.
    // If so, prepend an arbitrarily chosen character.
    // Change from original implementation, but should be safe.
    if (startWithIc("pg_", databaseName)) {
      databaseName = "x" + databaseName;
    }

    // Legacy: don't throw if the result is now empty.
    return databaseName;
  }
}
