package com.getmati.mati_sdk

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.media.MediaPlayer
import android.os.Build
import android.os.StatFs
import android.os.SystemClock
import android.os.storage.StorageManager
import android.view.TextureView
import android.view.View
import android.view.animation.AlphaAnimation
import android.view.animation.Animation
import androidx.core.app.ActivityCompat
import androidx.core.content.getSystemService
import java.io.IOException

internal inline fun Context.checkDiskSpace(requiredBytes: Long, action: (Boolean) -> Unit) {
    val available = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
        val storageManager = applicationContext.getSystemService<StorageManager>()!!
        val appSpecificInternalDirUuid = storageManager.getUuidForPath(cacheDir)
        val availableBytes = storageManager.getAllocatableBytes(appSpecificInternalDirUuid)
        if (availableBytes >= requiredBytes) {
            true
        } else {
            try {
                storageManager.allocateBytes(appSpecificInternalDirUuid, requiredBytes)
                true
            } catch (e: IOException) {
                false
            }
        }
    } else {
        val statFs = StatFs(cacheDir.path)
        statFs.availableBytes >= requiredBytes
    }
    action(available)
}

internal fun View.setSingleClickListener(freezePeriod: Long = 1000L, listener: (View) -> Unit) {
    var mLastClickTime = 0L
    setOnClickListener {
        if (SystemClock.elapsedRealtime() - mLastClickTime > freezePeriod) {
            mLastClickTime = SystemClock.elapsedRealtime()
            listener(it)
        }
    }
}

internal fun View.blink(duration: Long) {
    startAnimation(AlphaAnimation(0.0f, 1.0f).also {
        it.duration = duration
        it.startOffset = 20L
        it.repeatMode = Animation.REVERSE
        it.repeatCount = Animation.INFINITE
    })
}

internal fun Context.hasCameraPermission() = Build.VERSION.SDK_INT < Build.VERSION_CODES.M
        || ActivityCompat.checkSelfPermission(this, Manifest.permission.CAMERA) ==
        PackageManager.PERMISSION_GRANTED

internal fun Context.hasAudioPermission() = ActivityCompat.checkSelfPermission(this,
        Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED


internal fun String.isValidEmail() =
        isNotEmpty() && android.util.Patterns.EMAIL_ADDRESS.matcher(this).matches()

internal fun TextureView.centerCropMirrored(mp: MediaPlayer, mirrored: Boolean) {
    val horizontalMirroringFactor = if(mirrored) -1f else 1f
    val videoRatio = mp.videoWidth / mp.videoHeight.toFloat()
    val screenRatio = width / height.toFloat()
    val scaleXValue = videoRatio / screenRatio
    if (scaleXValue >= 1f) {
        scaleX = horizontalMirroringFactor * scaleXValue
    } else {
        scaleY = 1f / scaleXValue
        scaleX = horizontalMirroringFactor
    }
}


