package com.getmati.mati_sdk

import android.app.Activity
import android.content.Context
import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.util.AttributeSet
import android.view.LayoutInflater
import android.widget.TextView
import android.widget.Toast
import androidx.annotation.AttrRes
import androidx.annotation.ColorInt
import androidx.cardview.widget.CardView
import androidx.core.content.ContextCompat
import com.getmati.mati_sdk.ui.utils.Util


class MatiButton @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    @AttrRes defStyleAttr: Int = 0
) : CardView(context, attrs, defStyleAttr) {

    private var activity: Activity? = null
    private var clientId: String? = null
    private var flowId: String? = null
    private var metadata: Metadata? = null

    @ColorInt
    private val buttonColor: Int
    @ColorInt
    private val buttonTextColor: Int

    private val paint = Paint().apply {
        color = Color.WHITE
        style = Paint.Style.FILL
        flags = Paint.ANTI_ALIAS_FLAG
    }

    var text: CharSequence?
        get() {
            return mText.text
        }
        set(value) {
            mText.text = value
        }

    private val mText by lazy { findViewById<TextView>(R.id.label_text) }

    init {
        LayoutInflater.from(context).inflate(R.layout.frag_login_button, this, true)
        radius = resources.getDimensionPixelSize(R.dimen.mati_button_corner_radius).toFloat()
        minimumHeight = resources.getDimensionPixelSize(R.dimen.mati_button_min_height)

        with(context.theme.obtainStyledAttributes(attrs, R.styleable.MatiLoginButton, 0, 0)) {
            buttonColor = getColor(
                R.styleable.MatiLoginButton_color,
                ContextCompat.getColor(context, R.color.matiColorAccent)
            )
            buttonTextColor = getColor(
                R.styleable.MatiLoginButton_textColor,
                ContextCompat.getColor(context, R.color.matiPrimaryTextInverse)
            )

            setCardBackgroundColor(buttonColor)
            setTextColor(buttonTextColor)

            text = getString(R.styleable.MatiLoginButton_text)
                ?: context.getString(R.string.label_verify_me)


            recycle()

            setOnClickListener { _ ->
                activity?.let {
                    clientId?.let { clientId ->
                        metadata?.dataJson?.put(Metadata.KEY_BUTTON_COLOR, buttonColor)
                        metadata?.dataJson?.put(Metadata.KEY_BUTTON_TEXT_COLOR, buttonTextColor)
                        MatiSdk.startFlow(it, clientId, flowId, metadata)
                    }
                } ?: showMissingParamsError()
            }
        }
    }

    override fun onDraw(canvas: Canvas) {
        super.onDraw(canvas)
        canvas.drawCircle(
            Util.dpToPx(context, 80f) - 1.8f * height,
            height / 2f,
            1.8f * height,
            paint
        )
    }

    fun setTextColor(@ColorInt color: Int) {
        mText.setTextColor(color)
    }

    @JvmOverloads
    fun setParams(
        activity: Activity,
        clientId: String,
        flowId: String? = null,
        metadata: Metadata? = null
    ) {
        this.activity = activity
        this.clientId = clientId
        this.flowId = flowId
        this.metadata = metadata?.let {
            Metadata(
                it.dataJson.apply {
                    put(Metadata.KEY_BUTTON_COLOR, buttonColor)
                    put(Metadata.KEY_BUTTON_TEXT_COLOR, buttonTextColor)
                }.toString()
            )
        }
    }

    private fun showMissingParamsError() {
        Toast.makeText(context, "Params are not set.", Toast.LENGTH_SHORT).show()
    }
}