package com.getmati.mati_sdk.analytics.events

import com.getmati.mati_sdk.models.clean.CustomDoc.Companion.PREFIX_CUSTOM_DOCUMENT
import com.getmati.mati_sdk.models.clean.DrivingLicense
import com.getmati.mati_sdk.models.clean.DrivingLicense.Companion.DOC_ID_DRIVING_LICENSE
import com.getmati.mati_sdk.models.clean.NationalId
import com.getmati.mati_sdk.models.clean.NationalId.Companion.DOC_ID_NATIONAL_ID
import com.getmati.mati_sdk.models.clean.Passport
import com.getmati.mati_sdk.models.clean.Passport.Companion.DOC_ID_PASSPORT
import com.getmati.mati_sdk.models.clean.ProofOfResidency
import com.getmati.mati_sdk.models.clean.ProofOfResidency.Companion.DOC_ID_PROOF_OF_RESIDENCY
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.ui.VerificationStep
import org.json.JSONArray
import org.json.JSONObject

internal class Verification(
    val verificationState: VerificationState,
    val verificationStep: VerificationStep,
    val requiredDocuments: List<List<String>>,
    val requiredDocumentsCount: Int,
    val biometryType: String
) : AnalyticEvent("verification") {
    override fun getDescriptionJson(): JSONObject {
        return JSONObject().apply {
            put("requiredDocuments", requiredDocuments.transformRequiredDocuments())
            put("requiredDocumentsCount", requiredDocumentsCount)
            put("biometryType", biometryType)
            verificationState.putToJson(this)
            verificationStep.putToJson(this)
        }
    }
}

internal abstract class VerificationState {
    abstract fun putToJson(target: JSONObject)
}

internal class VerificationStarted : VerificationState() {
    override fun putToJson(target: JSONObject) {
        target.put("verificationState", "started")
    }
}

//internal class VerificationResumed: VerificationState(){
//    override fun putToJson(target: JSONObject) {
//        target.put("verificationState", "verificationResumed")
//    }
//}

internal class VerificationCanceled : VerificationState() {
    override fun putToJson(target: JSONObject) {
        target.put("verificationState", "canceled")
    }
}

internal class VerificationFailed(val errorMessage: String) : VerificationState() {
    override fun putToJson(target: JSONObject) {
        target.put("verificationState", JSONObject().apply {
            put("name", "failed")
            put("errorMessage", errorMessage)
        })
    }
}

internal abstract class Expired() : VerificationState() // TODO SHALL IT EXIST ON MOBILE -- ?

internal class VerificationPassed : VerificationState() {
    override fun putToJson(target: JSONObject) {
        target.put("verificationState", "passed")
    }
}

private fun List<List<String>>.transformRequiredDocuments(): JSONArray {
    return this.map { group ->
        if(group.size > 1) {
            group.map{ analyticsIdForDocId(it) }.joinToString()
        } else {
            "*${group.firstOrNull()?.let{analyticsIdForDocId(it)}}*"
        }
    }.let { groupsToString ->
        JSONArray().apply{
            groupsToString.forEach { put(it) }
        }
    }
}

private fun analyticsIdForDocId(id:String) = when (id) {
    DOC_ID_PASSPORT -> Passport.analyticsId
    DOC_ID_DRIVING_LICENSE -> DrivingLicense.analyticsId
    DOC_ID_NATIONAL_ID -> NationalId.analyticsId
    DOC_ID_PROOF_OF_RESIDENCY -> ProofOfResidency.analyticsId
    else -> if (id.startsWith(PREFIX_CUSTOM_DOCUMENT)) {
        CustomDoc.analyticsId
    }else {
        throw IllegalArgumentException("Unhandled document")
    }
}