package com.getmati.mati_sdk.managers.request_manager

import com.getmati.mati_sdk.BuildConfig
import com.getmati.mati_sdk.logger.LoggerFactory
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import io.ktor.client.*
import io.ktor.client.engine.*
import io.ktor.client.engine.android.*
import io.ktor.client.features.*
import io.ktor.client.features.json.*
import io.ktor.client.features.json.serializer.*
import io.ktor.client.features.logging.*
import io.ktor.client.request.*
import io.ktor.http.*
import kotlinx.serialization.json.Json as SerializationJson


internal class ApiRequestManager(private val prefetchDataHolder: PrefetchDataHolder) {

    companion object {
        private const val TIME_OUT = 20_000
        internal const val PREFIX_DEV_ENV_NAME = "dev-0"
        internal const val STAGING_ENV_NAME = "staging"
    }

    private val urls by lazy { defineUrls() }
    private val baseUrl by lazy { urls.baseUrl }
    private val uploadUrl by lazy { urls.uploadUrl }
    val webVerificationBaseUrl by lazy { urls.webVerificationBaseUrl }

    val client = HttpClient(engine()) {

        install(UserAgent) {
            agent = "MetaMap"
        }

        install(JsonFeature) {
            val json = SerializationJson {
                isLenient = true
                ignoreUnknownKeys = true
            }
            serializer = KotlinxSerializer(json)
            acceptContentTypes = acceptContentTypes + ContentType.Any
        }

        install(Logging) {
            logger = object : Logger {
                override fun log(message: String) {
                    LoggerFactory.logInfo(message)
                }
            }
            level = LogLevel.INFO
        }

        install(DefaultRequest) {
            header(
                "X-Mati-App",
                String.format("platform=android; version=%s", BuildConfig.VERSION_NAME)
            )
            if (prefetchDataHolder.hasPrefetchedData) {
                val accessToken = prefetchDataHolder.prefetchedData.accessToken
                header(HttpHeaders.Authorization, "Bearer $accessToken")
            }
            url {
                val isMedia = encodedPath.startsWith("/file") //TODO figure out better solution
                protocol = URLProtocol.HTTPS
                host = if (isMedia) uploadUrl else baseUrl
            }
        }
    }

    private fun engine() = Android.config {
        connectTimeout = TIME_OUT
        socketTimeout = TIME_OUT
    }

    private fun defineUrls() = when (val env = prefetchDataHolder.config?.env) {
        STAGING_ENV_NAME -> UrlsHolder(
            "api.stage.getmati.com",
            "media.stage.getmati.com",
            "product.staging.getmati.com"
        )
        null -> UrlsHolder("api.getmati.com", "media.getmati.com", "product.getmati.com")
        else -> if (env.startsWith(PREFIX_DEV_ENV_NAME)) {
            UrlsHolder("api.${env}.mati.io", "media.${env}.mati.io", "product.${env}.mati.io")
        } else {
            throw IllegalArgumentException("Unhandled environment")
        }
    }

    data class UrlsHolder(
        val baseUrl: String,
        val uploadUrl: String,
        val webVerificationBaseUrl: String
    )
}