package com.getmati.mati_sdk.managers.request_manager

import com.getmati.mati_sdk.models.api.response.ApiResponse
import io.ktor.client.*
import io.ktor.client.features.*
import io.ktor.client.request.*
import io.ktor.client.request.forms.*
import io.ktor.http.*
import io.ktor.http.content.*

internal suspend inline fun <reified T : Any> HttpClient.postCatching(
    urlString: String,
    block: HttpRequestBuilder.() -> Unit = {}
): ApiResponse<T> {
    return requestCatching(HttpMethod.Post, urlString, block)
}

internal suspend inline fun <reified T : Any> HttpClient.getCatching(
    urlString: String,
    block: HttpRequestBuilder.() -> Unit = {}
): ApiResponse<T> {
    return requestCatching(HttpMethod.Get, urlString, block)
}

internal suspend inline fun <reified T : Any> HttpClient.putCatching(
    urlString: String,
    block: HttpRequestBuilder.() -> Unit = {}
): ApiResponse<T> {
    return requestCatching(HttpMethod.Put, urlString, block)
}

suspend inline fun <reified T : Any> HttpClient.submitFormCatching(
    url: String,
    formData: List<PartData>,
    block: HttpRequestBuilder.() -> Unit = {}
): ApiResponse<T> = submitFormWithBinaryData(formData) {
    url(url)
    block()
    val result: Result<T> = runCatching {
        submitFormWithBinaryData(formData) {
            url(url)
            block()
        }
    }
    val apiResponse = result.mapCatching {
        ApiResponse.Success(it)
    }.getOrElse {
        if (it is ResponseException) {
            val responseCode = it.response.status.value
            val errorMessage = it.message
            ApiResponse.Error(responseCode, errorCode = "" /**TODO get error code **/, errorMessage = errorMessage)
        } else {
            ApiResponse.Error(-1)
        }
    }
    return apiResponse
}

private suspend inline fun <reified T : Any> HttpClient.requestCatching(
    method: HttpMethod,
    urlString: String,
    block: HttpRequestBuilder.() -> Unit = {}
): ApiResponse<T> {
    val result: Result<T> = runCatching {
        request {
            this.method = method
            url.takeFrom(urlString)
            block()
        }
    }
    val apiResponse = result.mapCatching {
        ApiResponse.Success(it)
    }.getOrElse {
        if (it is ResponseException) {
            val responseCode = it.response.status.value
            val errorMessage = it.message
            ApiResponse.Error(responseCode, errorCode = "", errorMessage = errorMessage)
        } else {
            ApiResponse.Error(-1)
        }
    }
    return apiResponse
}