package com.getmati.mati_sdk.mappers

import com.getmati.mati_sdk.models.clean.verification.VerificationFlow
import com.getmati.mati_sdk.models.socket.response.join_room.VerificationFlowResponse
import com.getmati.mati_sdk.ui.*
import com.getmati.mati_sdk.ui.web.BankAccountVerification
import com.getmati.mati_sdk.ui.web.PayrollAccountVerification
import com.getmati.mati_sdk.ui.web.WorkAccountVerification

private const val VALUE_BIOMETRICS_LIVENESS = "liveness"
private const val VALUE_BIOMETRICS_SELFIE = "selfie"
private const val VALUE_BIOMETRICS_VOICE_LIVENESS = "voice+liveness"
private const val SMS_OPTIONAL = "sms+optional"
private const val SMS = "sms"
private const val NONE = "none"

internal fun VerificationFlowResponse.toVerificationFlow(): VerificationFlow {

    val verificationFlowResponse = this
    return VerificationFlow(
        id = verificationFlowResponse.id ?: "NONE",
        name = verificationFlowResponse.name ?: "NONE",
        verificationSteps = transformFlowDataToSteps(verificationFlowResponse),
        denyUploadsFromMobileGallery = verificationFlowResponse.denyUploadsFromMobileGallery
            ?: false,
        pinnedCountries = verificationFlowResponse.pinnedCountries,
        supportedCountries = verificationFlowResponse.supportedCountries ?: emptyList(),
        logoUrl = verificationFlowResponse.logoUrl,
        ipValidation = verificationFlowResponse.ipValidationResponse?.toIpValidation()
    )
}

private fun transformFlowDataToSteps(verificationFlow: VerificationFlowResponse): List<VerificationStep> {
    val result = mutableListOf<VerificationStep>()

    val verificationPatterns = verificationFlow.verificationPatternsResponse

    if (verificationPatterns?.electronicSignatureDocumentValidation == true) {
        val eSignVerificationStep =
            verificationFlow.electronicSignatureResponse?.toESignVerificationStep()
        eSignVerificationStep?.let {
            result.add(eSignVerificationStep)
        }
    }

    verificationFlow.toDocuments().forEachIndexed { index, docs ->
        result.add(DocumentVerificationStep(docs, index))
    }

    verificationPatterns?.biometrics.let { biometry ->
        when (biometry) {
            VALUE_BIOMETRICS_SELFIE -> {
                result.add(BiometryUpload(BiometryType.SELFIE))
            }
            VALUE_BIOMETRICS_LIVENESS -> {
                result.add(BiometryUpload(BiometryType.SELFIE_VIDEO))
            }
            VALUE_BIOMETRICS_VOICE_LIVENESS -> {
                result.add(BiometryUpload(BiometryType.VOICE_LIVENESS))
                result.add(BiometryUpload(BiometryType.SELFIE_VIDEO))
            }
            else -> {
            }
        }
    }
    verificationPatterns?.phoneOwnershipValidation.let {
        when (it) {
            SMS -> result.add(SmsUpload(false))
            SMS_OPTIONAL -> result.add(SmsUpload(true))
            else -> {
            }
        }
    }
    verificationPatterns?.emailOwnershipValidation.let {
        if (it != NONE) {
            result.add(
                EmailVerification(
                    it == EmailVerification.KEY_OPTIONAL,
                    5,
                    30000
                )
            )
        }
    }
    if (verificationPatterns?.bankAccountRetrieval == true) {
        val webContainerStep =
            verificationFlow.toWebVerification(BankAccountVerification)
        result.add(webContainerStep)
    }
    if (verificationPatterns?.payrollAccountRetrieval == true) {
        val webContainerStep =
            verificationFlow.toWebVerification(PayrollAccountVerification)
        result.add(webContainerStep)
    }
    if (verificationPatterns?.workAccountRetrieval == true) {
        val webContainerStep =
            verificationFlow.toWebVerification(WorkAccountVerification)
        result.add(webContainerStep)
    }

    return result
}