package com.getmati.mati_sdk.mati_navigation

import android.app.Activity
import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import androidx.core.app.ActivityCompat
import androidx.core.os.bundleOf
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.navigation.Navigation
import androidx.savedstate.SavedStateRegistry
import androidx.savedstate.SavedStateRegistryOwner
import com.getmati.mati_sdk.MatiSdk
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.*
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.DocPage
import com.getmati.mati_sdk.models.clean.input.InputError
import com.getmati.mati_sdk.models.clean.verification.VerificationFlow
import com.getmati.mati_sdk.models.clean.verification.biometryType
import com.getmati.mati_sdk.models.clean.verification.documentUploadSteps
import com.getmati.mati_sdk.ui.*
import com.getmati.mati_sdk.ui.common.ExitFragment
import com.getmati.mati_sdk.ui.common.KYCBaseFragment.Companion.ARG_NEW_STEP_STARTED
import com.getmati.mati_sdk.ui.common.VerificationSuccessFragment
import com.getmati.mati_sdk.ui.doc_hint.DocumentHintFragment
import com.getmati.mati_sdk.ui.document.CountriesRepo
import com.getmati.mati_sdk.ui.document.SelectCountryFragment
import com.getmati.mati_sdk.ui.document.SelectDocumentFragment
import com.getmati.mati_sdk.ui.email.email_submission.EmailSubmissionFragment
import com.getmati.mati_sdk.ui.esign.ESignHostFragment
import com.getmati.mati_sdk.ui.iprestrictions.IpCountryRestrictedFragment
import com.getmati.mati_sdk.ui.iprestrictions.VpnDetectedFragment
import com.getmati.mati_sdk.ui.verification.VerificationActivity
import com.getmati.mati_sdk.ui.verification.VerificationVm
import com.getmati.mati_sdk.ui.liveness.LivenessFragment
import com.getmati.mati_sdk.ui.liveness.VoiceLivenessFragment
import com.getmati.mati_sdk.ui.phonevalidation.PhoneInputFragment
import com.getmati.mati_sdk.ui.selfie.SelfieHintFragment
import com.getmati.mati_sdk.ui.web.WebVerificationFragment
import org.koin.core.component.KoinComponent
import org.koin.core.component.inject

internal class MatiNavigation(
    val activity: VerificationActivity, val verificationVm: VerificationVm,
    private val registryOwner: SavedStateRegistryOwner
) : SavedStateRegistry.SavedStateProvider, KoinComponent {

    private val countriesRepo by inject<CountriesRepo>()
    private var currentVerificationStepIdx: Int

    init {
        currentVerificationStepIdx = -1

        registryOwner.lifecycle.addObserver(LifecycleEventObserver { _, event ->
            if (event == Lifecycle.Event.ON_CREATE) {
                val registry = registryOwner.savedStateRegistry
                registry.registerSavedStateProvider(PROVIDER, this)
                registry.consumeRestoredStateForKey(PROVIDER)?.run {
                    currentVerificationStepIdx = getInt(ARG_VERIFICATION_STEP_IDX)
                }
            }
        })
    }

    private val verificationFlow: VerificationFlow
        get() {
            return verificationVm.verificationFlow
        }

    private val restrictionError: InputError?
        get() {
            return verificationVm.inputsFlow.value.firstOrNull { it.id == "connection-data" }?.error
        }

    fun checkPermission(permission: String): Boolean {
        return Build.VERSION.SDK_INT > Build.VERSION_CODES.M &&
                ActivityCompat.checkSelfPermission(
                    activity,
                    permission
                ) == PackageManager.PERMISSION_GRANTED
    }

    private val navController by lazy {
        Navigation.findNavController(
            activity,
            R.id.nav_host_fragment
        )
    }
    private val supportedCountries =
        countriesRepo.filterSupported(verificationFlow.supportedCountries)

    fun startVerification() {
        track(
            Verification(
                VerificationStarted(),
                verificationFlow.verificationSteps.first(),
                verificationFlow.documentUploadSteps.map { group -> group.map { it.id } },
                verificationFlow.verificationSteps.size,
                verificationFlow.biometryType
            )
        )

        when (restrictionError?.code) {
            "connectionData.restricted" -> navigateTo(
                IpCountryRestrictedFragment.destination(
                    restrictionError!!.isCritical
                )
            )
            "connectionData.vpnDetected" -> navigateTo(
                VpnDetectedFragment.destination(
                    restrictionError!!.isCritical
                )
            )
            else -> openVerificationStep(0)
        }
    }

    fun openVerificationStep(verificationStepIdx: Int) {
        currentVerificationStepIdx = verificationStepIdx

        val nextStepDestination =
            when (val verificationStep = verificationFlow.verificationSteps[verificationStepIdx]) {
                is ESignVerificationStep -> {
                    ESignHostFragment.destination()
                }
                is DocumentVerificationStep -> {
                    if (verificationStep.acceptableDocuments.size == 1) {
                        val doc = verificationStep.acceptableDocuments.first()
                        val skippable = doc is CustomDoc && doc.skippable
                        when (supportedCountries.size) {
                            1 -> {
                                doc.country = supportedCountries[0]
                                if (countriesRepo.shouldShowSelectRegion(
                                        doc.country!!.code,
                                        doc
                                    )
                                ) {
                                    SelectCountryFragment.destination(
                                        doc,
                                        verificationStep.group,
                                        skippable
                                    )
                                } else {
                                    DocumentHintFragment.destination(
                                        DocPage(doc, 1),
                                        verificationStep.group,
                                        skippable
                                    )
                                }
                            }
                            else -> SelectCountryFragment.destination(
                                doc,
                                verificationStep.group,
                                skippable
                            )
                        }

                    } else {
                        SelectDocumentFragment.destination(verificationStep)
                    }
                }
                is BiometryUpload -> {
                    track(BiometryUploadEvent(Started(), verificationStep.biometryType))
                    when (verificationStep.biometryType) {
                        BiometryType.SELFIE_VIDEO -> LivenessFragment.destination()
                        BiometryType.VOICE_LIVENESS -> VoiceLivenessFragment.destination()
                        BiometryType.SELFIE -> SelfieHintFragment.destination()
                        else -> ExitFragment.destination() // Something went wrong
                    }
                }
                is SmsUpload -> {
                    PhoneInputFragment.destination(
                        verificationStep.optional,
                        verificationStep.timeoutMs
                    )
                }
                is EmailVerification -> {
                    verificationStep.run {
                        EmailSubmissionFragment.destination(optional, attemptLimit, coolDown)
                    }
                }

                is WebVerificationStep -> {
                    WebVerificationFragment.destination(verificationStep.type)
                }
                else -> ExitFragment.destination() // Something went wrong
            }

        navigateTo(nextStepDestination.apply { args.putBoolean(ARG_NEW_STEP_STARTED, true) })
    }

    fun openNextStep() {
        if (currentVerificationStepIdx == verificationFlow.verificationSteps.lastIndex) {
            navigateTo(
                VerificationSuccessFragment.destination()
                    .apply { args.putBoolean(ARG_NEW_STEP_STARTED, true) })
        } else {
            openVerificationStep(currentVerificationStepIdx + 1)
        }
    }

    fun navigateTo(destination: MatiDestination) {
        navController.navigate(destination.actionId, destination.args)
    }

    fun back() {
        navController.popBackStack()
    }

    fun cancelVerification() {
        verificationFlow.apply {
            if (verificationSteps.isNotEmpty()) {
                track(
                    Verification(
                        VerificationCanceled(),
                        verificationSteps[kotlin.math.max(currentVerificationStepIdx, 0)],
                        documentUploadSteps.map { group -> group.map { it.id } },
                        verificationSteps.size,
                        biometryType
                    )
                )
            }
        }
        finishWithResult(Activity.RESULT_CANCELED)
    }

//    fun failVerificaiton(errorMessage: String) { // TODO Need to be called on crash @Hayk Kerobyan
//        track(Verification(VerificationFailed(errorMessage), verificationSteps[verificationStepIdx]))
//        activity.setResult(Activity.RESULT_CANCELED, activity.mData)
//        activity.finish()
//    }

    fun finishVerification() {
        track(
            Verification(
                VerificationPassed(),
                verificationFlow.verificationSteps.last(),
                verificationFlow.documentUploadSteps.map { group -> group.map { it.id } },
                verificationFlow.verificationSteps.size,
                verificationFlow.biometryType
            )
        )
        finishWithResult(Activity.RESULT_OK)
    }

    private fun finishWithResult(result: Int) {
        activity.setResult(result, Intent().apply {
            putExtra(MatiSdk.ARG_IDENTITY_ID, verificationVm.prefetchedData.identityId)
            putExtra(MatiSdk.ARG_VERIFICATION_ID, verificationVm.prefetchedData.verificationId)
        })
        activity.finish()
    }

    override fun saveState(): Bundle {
        return bundleOf(ARG_VERIFICATION_STEP_IDX to currentVerificationStepIdx)
    }

    companion object {
        private const val PROVIDER = "MatiNavigation"
        private const val ARG_VERIFICATION_STEP_IDX = "ARG_VERIFICATION_STEP_IDX"
    }
}

internal data class MatiDestination(val actionId: Int, val args: Bundle)