package com.getmati.mati_sdk.network

import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.models.clean.verification.VerificationType
import com.getmati.mati_sdk.models.clean.input.Input
import com.getmati.mati_sdk.models.clean.input.InputStatus
import com.getmati.mati_sdk.server.SocketManager
import com.getmati.mati_sdk.models.api.response.ApiResponse
import com.getmati.mati_sdk.server.verification.StepVerificationResult
import com.getmati.mati_sdk.server.verification.VerificationError
import com.getmati.mati_sdk.server.verification.VerificationResultMapper
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.flow.*
import org.json.JSONObject

internal class NetManager(private val socketManager: SocketManager) {

    companion object {
        private const val ITEMS_TO_DROP_CURRENT_VALUE = 1
    }

    suspend fun <T> processVerification(
        timeout: Long = 15000L,
        verificationType: VerificationType,
        mapper: VerificationResultMapper<T>,
        group: Int? = null,
        isSkipStep: Boolean = false,
        request: suspend () -> ApiResponse<*>
    ): StepVerificationResult<T> {
        val channel = Channel<StepVerificationResult<T>>()
        val job = Job()
        coroutineScope {
            launch(Dispatchers.IO + job) {
                delay(timeout)
                if (!channel.isClosedForSend)
                    channel.send(
                        StepVerificationResult(
                            error = VerificationError(MediaVerificationError.OTHER)
                        )
                    )
            }
            launch(Dispatchers.IO + job) {
                socketManager.verificationResult
                    .drop(ITEMS_TO_DROP_CURRENT_VALUE)
                    .filterNotNull()
                    .filter { getInputId(it, isSkipStep) == verificationType.id }
                    .collect {
                        if (!channel.isClosedForSend)
                            channel.send(mapper.inputProcessedMapper(it))
                    }
            }
            launch(Dispatchers.IO + job) {
                val response = request()
                if (response is ApiResponse.Success) {
                    socketManager.verificationHistory
                        .drop(ITEMS_TO_DROP_CURRENT_VALUE)
                        .filterNotNull()
                        .map { retrieveVerificationObject(it, verificationType, group) }
                        .filterNotNull()
                        .filter { it.status?.code != InputStatus.IN_PROGRESS.code }
                        .collect {
                            if (!channel.isClosedForSend)
                                channel.send(mapper.inputJoinRoomMapper(it))
                        }
                } else if (!channel.isClosedForSend) {
                    channel.send(
                        StepVerificationResult(
                            error = VerificationError(
                                MediaVerificationError.OTHER
                            )
                        )
                    )
                }
            }
        }
        val result = channel.consumeAsFlow().first()
        channel.close()
        job.cancel()
        return result
    }

    private fun getInputId(
        jsonObject: JSONObject,
        isSkipStep: Boolean
    ) = if (isSkipStep) {
        jsonObject.getString("id")
    } else {
        jsonObject.optJSONObject("data")?.getString("inputId")
    }

    private fun retrieveVerificationObject(
        inputs: List<Input>,
        verificationType: VerificationType,
        group: Int? = null
    ): Input? {
        return inputs.firstOrNull {
            it.id == verificationType.id && group == it.group
        }
    }
}