package com.getmati.mati_sdk.sentry.io.sentry;

import com.getmati.mati_sdk.sentry.io.sentry.protocol.SdkVersion;
import com.getmati.mati_sdk.sentry.io.sentry.protocol.SentryId;
import com.getmati.mati_sdk.sentry.io.sentry.util.Objects;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public final class SentryEnvelope {

  // types: session_batch, session, event, attachment
  // an envelope cannot have 2 events, but eg multiple sessions
  private final @NotNull
  SentryEnvelopeHeader header;
  private final @NotNull Iterable<SentryEnvelopeItem> items;

  public @NotNull Iterable<SentryEnvelopeItem> getItems() {
    return items;
  }

  public @NotNull
  SentryEnvelopeHeader getHeader() {
    return header;
  }

  public SentryEnvelope(
      final @NotNull SentryEnvelopeHeader header,
      final @NotNull Iterable<SentryEnvelopeItem> items) {
    this.header = Objects.requireNonNull(header, "SentryEnvelopeHeader is required.");
    this.items = Objects.requireNonNull(items, "SentryEnvelope items are required.");
  }

  public SentryEnvelope(
      final @Nullable SentryId eventId,
      final @Nullable SdkVersion sdkVersion,
      final @NotNull Iterable<SentryEnvelopeItem> items) {
    header = new SentryEnvelopeHeader(eventId, sdkVersion);
    this.items = Objects.requireNonNull(items, "SentryEnvelope items are required.");
  }

  public SentryEnvelope(
      final @Nullable SentryId eventId,
      final @Nullable SdkVersion sdkVersion,
      final @NotNull SentryEnvelopeItem item) {
    Objects.requireNonNull(item, "SentryEnvelopeItem is required.");

    header = new SentryEnvelopeHeader(eventId, sdkVersion);
    final List<SentryEnvelopeItem> items = new ArrayList<>(1);
    items.add(item);
    this.items = items;
  }

  public static @NotNull
  SentryEnvelope from(
      final @NotNull ISerializer serializer,
      final @NotNull Session session,
      final @Nullable SdkVersion sdkVersion)
      throws IOException {
    Objects.requireNonNull(serializer, "Serializer is required.");
    Objects.requireNonNull(session, "session is required.");

    return new SentryEnvelope(
        null, sdkVersion, SentryEnvelopeItem.fromSession(serializer, session));
  }

  public static @NotNull
  SentryEnvelope from(
      final @NotNull ISerializer serializer,
      final @NotNull SentryBaseEvent event,
      final @Nullable SdkVersion sdkVersion)
      throws IOException {
    Objects.requireNonNull(serializer, "Serializer is required.");
    Objects.requireNonNull(event, "item is required.");

    return new SentryEnvelope(
        event.getEventId(), sdkVersion, SentryEnvelopeItem.fromEvent(serializer, event));
  }
}
