package com.getmati.mati_sdk.server

import com.getmati.mati_sdk.logger.LoggerFactory
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import com.getmati.mati_sdk.models.clean.SocketEvent
import com.getmati.mati_sdk.models.clean.input.Input
import com.getmati.mati_sdk.models.socket.response.countries.CountryResponse
import com.getmati.mati_sdk.models.socket.response.join_room.JoinRoomResponse
import com.getmati.mati_sdk.models.socket.response.join_room.VerificationResultResponse
import com.getmati.mati_sdk.managers.request_manager.ApiRequestManager
import com.getmati.mati_sdk.mappers.*
import com.getmati.mati_sdk.mappers.toCountries
import com.getmati.mati_sdk.mappers.toInputList
import com.getmati.mati_sdk.mappers.toVerificationFlow
import com.getmati.mati_sdk.mappers.toVerificationResult
import io.socket.client.Manager
import io.socket.client.Socket
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.decodeFromJsonElement
import org.json.JSONObject
import java.net.URI

internal const val DATA = "data"
internal const val ROOM = "room"
internal const val JOIN_ROOM = "joinRoom"
internal const val JOIN_ROOM_COMPLETED = "joinRoom.completed"
internal const val COUNTRY_LIST = "countries.list"
internal const val VERIFICATION_PROCESSED = "verification.input.processed"

internal class SocketManager(private val prefetchDataHolder: PrefetchDataHolder, private val jsonBuilder: Json) {

    private val prefetchedData by lazy { prefetchDataHolder.prefetchedData }
    private val accessToken by lazy { prefetchedData.accessToken }
    private val verificationId by lazy { prefetchedData.verificationId }
    private val socketUrl by lazy { constructSocketUrl(prefetchedData.config?.env) }
    private val mutableEventFlow = MutableStateFlow<SocketEvent>(SocketEvent.DummySocketEvent)

    val verificationResult = MutableStateFlow<JSONObject?>(null)
    val verificationHistory = MutableStateFlow<List<Input>?>(null)
    val eventFlow: StateFlow<SocketEvent> = mutableEventFlow

    private val flowSocket by lazy {
        Manager(URI(socketUrl), Manager.Options().apply {
            path = "/ws/socket.io"
            secure = true
            query = "token=$accessToken"
            policyPort = 843
            transports = arrayOf("websocket")
            this.reconnection = true
        }).socket("/verification-flow")
    }

    private val eventSocket by lazy {
        Manager(URI(socketUrl), Manager.Options().apply {
            path = "/ws/socket.io"
            secure = true
            query = "token=$accessToken"
            policyPort = 843
            transports = arrayOf("websocket")
            this.reconnection = true
        }).socket("/sdk-stats")
    }

    fun connect() {
        flowSocket.on(JOIN_ROOM_COMPLETED) { event ->
            event.firstOrNull()?.let {
                LoggerFactory.logInfo("$TAG  ${(it as JSONObject).toString(2)}")
                val joinRoomData = it.optJSONObject(DATA)!!
                val json = jsonBuilder.parseToJsonElement(joinRoomData.toString())
                val joinRoomResponse = jsonBuilder.decodeFromJsonElement<JoinRoomResponse>(json)
                val inputs = joinRoomResponse.inputs?.toInputList()
                val verificationFlow = joinRoomResponse.verificationFlowResponse.toVerificationFlow()
                val webContainerConfig = joinRoomResponse.webContainerConfigs.toWebContainerConfigList()
                mutableEventFlow.value = SocketEvent.RoomJoined(inputs ?: emptyList(), verificationFlow, webContainerConfig)
                verificationHistory.value = inputs
            }
        }
        flowSocket.on(VERIFICATION_PROCESSED) { event ->
            event.firstOrNull()?.let {
                LoggerFactory.logInfo("$TAG  ${(it as JSONObject).toString(2)}")
                val json = jsonBuilder.parseToJsonElement(it.toString())
                val verificationResultResponse = jsonBuilder.decodeFromJsonElement<VerificationResultResponse>(json)
                val verification = verificationResultResponse.toVerificationResult()
                mutableEventFlow.value = SocketEvent.VerificationResultEvent(verification)
                verificationResult.value = it
            }
        }
        flowSocket.on(Socket.EVENT_CONNECT) {
            flowSocket.emit(JOIN_ROOM, JSONObject().apply { put(ROOM, verificationId) })
        }

        flowSocket.on(COUNTRY_LIST) { event ->
            event.firstOrNull()?.let {
                LoggerFactory.logInfo("$TAG  ${(it as JSONObject).toString(2)}")
                val json = jsonBuilder.parseToJsonElement(it.getJSONArray(DATA).toString())
                val countriesResponse = jsonBuilder.decodeFromJsonElement<List<CountryResponse>>(json)
                val countries = countriesResponse.toCountries()
                mutableEventFlow.value = SocketEvent.CountriesReceived(countries)
            }
        }

        eventSocket.connect()
        flowSocket.connect()
    }

    fun emit(event: String, args: JSONObject) {
        LoggerFactory.logInfo("$TAG  $event ${args.toString(2)}")
        flowSocket.emit(event, args)
    }

    fun emitAnalytics(event: String, args: JSONObject) {
        LoggerFactory.logInfo("$TAG $event ${args.toString(2)}")
        eventSocket.emit(event, args)
    }

    fun disconnect() {
        if (!prefetchDataHolder.hasPrefetchedData) {
            return
        }
        flowSocket.disconnect()
        flowSocket.off()

        eventSocket.disconnect()
        eventSocket.off()
    }

    fun connected() = flowSocket.connected() && eventSocket.connected()

    private fun constructSocketUrl(env: String?) = when (env) {
        null -> "wss://api.getmati.com"
        ApiRequestManager.STAGING_ENV_NAME -> "wss://api.stage.getmati.com"
        else -> if (env.startsWith(ApiRequestManager.PREFIX_DEV_ENV_NAME))
            "wss://api.$env.mati.io"
        else throw IllegalArgumentException("Unhandled environment")
    }

    companion object {
        const val TAG = "SocketManager"
    }
}