package com.getmati.mati_sdk.ui.camera

import android.Manifest
import android.annotation.SuppressLint
import android.util.Log
import android.widget.*
import androidx.activity.result.contract.ActivityResultContracts
import androidx.annotation.LayoutRes
import androidx.camera.core.AspectRatio
import androidx.camera.core.CameraSelector
import androidx.camera.core.Preview
import androidx.camera.lifecycle.ProcessCameraProvider
import androidx.camera.video.*
import androidx.camera.view.PreviewView
import androidx.concurrent.futures.await
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.core.util.Consumer
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.hasAudioPermission
import com.getmati.mati_sdk.hasCameraPermission
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import java.io.File
import java.text.SimpleDateFormat
import java.util.*

internal abstract class VideoCameraFragment(@LayoutRes layoutId: Int) : KYCBaseFragment(layoutId),
    CameraPermissionFragment {

    protected abstract var lensFacing: Int
    protected abstract val audioEnabled: Boolean
    protected abstract val maxDuration: Long

    protected abstract val previewView: PreviewView

    private lateinit var videoCapture: VideoCapture<Recorder>
    private var activeRecording: ActiveRecording? = null
    private var recordingState: VideoRecordEvent? = null

    private val mainThreadExecutor by lazy { ContextCompat.getMainExecutor(requireContext()) }
    private var enumerationDeferred: Deferred<Unit>? = null

    private var timerJob: Job? = null

    var cameraProvider: ProcessCameraProvider? = null

    private val _state = MutableStateFlow<State>(State.Initial)
    protected val state :StateFlow<State> = _state
    protected fun dropState(){
        _state.value = State.Preview
    }

    protected fun flipCamera(){
        lensFacing = when(lensFacing) {
            CameraSelector.LENS_FACING_FRONT -> CameraSelector.LENS_FACING_BACK
            else -> CameraSelector.LENS_FACING_FRONT
        }
        _state.value = State.Initial
        checkPermissionAndOpenCamera()
    }

    private val cameraPermissionResult = registerForActivityResult(
        ActivityResultContracts.RequestMultiplePermissions()
    ) { permissions ->
        val missingPermissions = ArrayList<String>()
        permissions.entries.forEach {
            if (!it.value) {
                missingPermissions.add(it.key)
            }
        }
        if (missingPermissions.isEmpty()) {
            if (requireContext().hasCameraPermission()
                && !audioEnabled || requireContext().hasAudioPermission()
            ) {
                bindCaptureUsecase()
            }
        } else {
            for (i in missingPermissions.indices) {
                val permission = missingPermissions[i]
                if (!ActivityCompat.shouldShowRequestPermissionRationale(
                        verificationActivity!!,
                        permission
                    )
                ) {
                    onPermissionPermanentlyDenied(permission)
                    return@registerForActivityResult
                }
            }
            onPermissionDenied(*missingPermissions.toTypedArray())
        }
    }

    override fun checkPermissionAndOpenCamera() {
        val context = requireContext()
        val missingPermissions = ArrayList<String>()
        if (!context.hasCameraPermission()) missingPermissions.add(Manifest.permission.CAMERA)
        if (audioEnabled && !context.hasAudioPermission()) missingPermissions.add(Manifest.permission.RECORD_AUDIO)
        if (missingPermissions.isEmpty()) {
            bindCaptureUsecase()
        } else {
            cameraPermissionResult.launch(*missingPermissions.toTypedArray())
        }
    }

    /**
     *   Always bind preview + video capture use case combinations in this sample
     *   (VideoCapture can work on its own).
     */
    private fun bindCaptureUsecase() {
        viewLifecycleOwner.lifecycleScope.launch {
            cameraProvider = ProcessCameraProvider.getInstance(requireContext()).await()

            val lens = cameraProvider!!.defineLens(lensFacing)

            if(lens == MISSING_LENS_VALUE){
                _state.value = State.CameraIsAbsent
                cameraProvider!!.unbindAll()
                _state.value = State.Initial
                return@launch
            }

            val cameraSelector = CameraSelector.Builder()
                .requireLensFacing(lens)
                .build()
            val preview = Preview.Builder().setTargetAspectRatio(DEFAULT_ASPECT_RATIO)
                .build().apply {
                    setSurfaceProvider(previewView.surfaceProvider)
                }

            // create the user required QualitySelector (video resolution): we know this is
            // supported, a valid qualitySelector will be created.
            val qualitySelector = QualitySelector.of(QualitySelector.QUALITY_LOWEST)

            // build a recorder, which can:
            //   - record video/audio to MediaStore(only use here), File, ParcelFileDescriptor
            //   - be used create recording(s) (the recording performs recording)
            val recorder = Recorder.Builder()
                .setQualitySelector(qualitySelector)
                .build()
            videoCapture = VideoCapture.withOutput(recorder)

            try {
                cameraProvider!!.unbindAll()
                cameraProvider!!.bindToLifecycle(
                    requireParentFragment(),
                    cameraSelector,
                    videoCapture,
                    preview
                )
                _state.value = State.Preview
            } catch (exc: Exception) {
                Log.e(TAG, "Use case binding failed", exc)
                _state.value = State.Initial
                checkPermissionAndOpenCamera()
            }
        }
    }

    /**
     * Kick start the video recording
     *   - config Recorder to capture to MediaStoreOutput
     *   - register RecordEvent Listener
     *   - apply audio request from user
     *   - start recording!
     * After this function, user could start/pause/resume/stop recording and application listens
     * to VideoRecordEvent for the current recording status.
     */
    @SuppressLint("MissingPermission")
    private fun startRecording() {
        // create MediaStoreOutputOptions for our recorder: resulting our recording!
        val name = SimpleDateFormat(FILENAME_FORMAT, Locale.US)
            .format(System.currentTimeMillis()) + ".mp4"

        val outputOptions = FileOutputOptions.Builder(File("${requireContext().cacheDir}/$name"))
            .build()

        // configure Recorder and Start recording to the mediaStoreOutput.
        activeRecording =
            videoCapture.output.prepareRecording(requireActivity(), outputOptions)
                .withEventListener(
                    mainThreadExecutor,
                    captureListener
                )
                .apply { if (audioEnabled) withAudioEnabled() }
                .start()

        setUpTimer()
        _state.value = State.Recording
        Log.i(TAG, "Recording started")
    }

    /**
     * CaptureEvent listener.
     */
    private val captureListener = Consumer<VideoRecordEvent> { event ->
        // cache the recording state
        if (event !is VideoRecordEvent.Status)
            recordingState = event

        if (event is VideoRecordEvent.Finalize) {
            _state.value = State.Recorded(event)
            cancelTimer()
        }
    }


    protected fun startRec() {
        if (recordingState == null || recordingState is VideoRecordEvent.Finalize) {
            startRecording()
        }
    }

    protected fun stopRec() {
        if (activeRecording == null
            || recordingState==null
            ||recordingState is VideoRecordEvent.Finalize) {
            return
        }

        val recording = activeRecording
        if (recording != null) {
            recording.stop()
            activeRecording = null
        }
    }

    // system functions starts
    override fun onResume() {
        super.onResume()
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
            if (enumerationDeferred != null) {
                enumerationDeferred!!.await()
                enumerationDeferred = null
            }
            _state.value = State.Initial
            checkPermissionAndOpenCamera()
        }
    }

    override fun onPause() {
        super.onPause()
        cameraProvider?.unbindAll()
        cameraProvider = null
    }

    private fun setUpTimer() {
        timerJob = viewLifecycleOwner.lifecycleScope.launchWhenStarted {
            delay(maxDuration)
            _state.value = State.MaxDurationReached
        }
    }

    private fun cancelTimer() {
        timerJob?.cancel()
        timerJob = null
    }


    sealed class State {
        object Initial : State() //waiting for permissions
        object Preview : State() //permissions granted, camera is open
        object Recording : State()
        data class Recorded(val event: VideoRecordEvent.Finalize) : State()
        object CameraIsAbsent : State()
        object MaxDurationReached : State()
    }

    companion object {
        // default QualitySelector if no input from UI
        private const val DEFAULT_ASPECT_RATIO = AspectRatio.RATIO_4_3
        private val TAG: String = VideoCameraFragment::class.java.simpleName
        private const val FILENAME_FORMAT = "yyyy-MM-dd-HH-mm-ss-SSS"
    }
}

