package com.getmati.mati_sdk.ui.common

import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import androidx.activity.OnBackPressedCallback
import androidx.annotation.LayoutRes
import androidx.fragment.app.Fragment
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Navigation
import com.getmati.mati_sdk.analytics.events.NavigationState
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.logger.LoggerFactory
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.models.clean.verification.VerificationFlow
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.verification.VerificationActivity
import com.getmati.mati_sdk.ui.verification.VerificationActivity.Companion.MIN_REQUIRED_BYTES_FOR_VIDEO
import com.getmati.mati_sdk.ui.verification.VerificationVm
import com.getmati.mati_sdk.ui.utils.BitmapTransformation
import com.getmati.mati_sdk.ui.utils.PdfToImageTransformation
import com.getmati.mati_sdk.ui.utils.findPrimaries
import com.getmati.mati_sdk.ui.utils.setUpMainActionButton
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

internal abstract class KYCBaseFragment(@LayoutRes layoutId: Int) : Fragment(layoutId) {
    abstract val screenName: String
    protected val verificationActivity: VerificationActivity?
        get() {
            return context as VerificationActivity?
        }
    protected val navigation by lazy { verificationActivity!!.navigation }
    protected val verificationVm: VerificationVm
        get() = verificationActivity!!.verificationVm

    protected val verificationFlow: VerificationFlow
        get() {
            return verificationVm.verificationFlow
        }

    protected val newStepStarted by lazy {
        arguments?.getBoolean(ARG_NEW_STEP_STARTED, false) ?: false
    }

    open fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.LIGHT)
        toolbar.setBackImageVisibile(!newStepStarted)
        toolbar.setCloseImageVisibile(true)
        toolbar.setChooseLanguageVisible(false)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        track(
            Navigation(
                screenName, if (savedInstanceState == null) {
                    NavigationState.OPENED
                } else {
                    NavigationState.REOPENED
                }
            )
        )
    }

    protected val blockBackToPreviousStep = object : OnBackPressedCallback(true) {
        override fun handleOnBackPressed() {
            if (newStepStarted) {
                navigation.navigateTo(ExitFragment.destination())
            } else {
                navigation.back()
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        LoggerFactory.logInfo("$screenName screen opened")
        configureToolbar(verificationActivity!!.toolbar)
        (view as? ViewGroup)?.findPrimaries()?.forEach {
            it.setUpMainActionButton(verificationVm.prefetchedData.config)
        }

        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            blockBackToPreviousStep
        )
    }

    private val MIN_REQUIRED_BYTES = 1024 * 1024 * 150L //150 MB

    protected fun handleFullDiskSpaceOrElse(
        requiredBytes: Long = MIN_REQUIRED_BYTES_FOR_VIDEO,
        action: () -> Unit
    ) =
        verificationActivity?.handleFullDiskSpaceOrElse(requiredBytes, action) // TODO Move somewhere


    internal suspend fun showErrorIfFailure(pdfToImageTransformation: PdfToImageTransformation): Boolean {
        val mediaVerificationError = when (pdfToImageTransformation) {
            is PdfToImageTransformation.PdfTransformationError ->
                MediaVerificationError.DOCUMENT_PDF_TRANSFORMATION_FAILURE
            //No Failure or Non pdf file
            else -> return false
        }
        withContext(Dispatchers.Main) {
            mediaVerificationError.run {
                navigation.navigateTo(
                    BaseErrorFragment.destination(prepareMediaErrorScreenData(title = getString(title), primaryCTALabel = getString(R.string.label_try_again)))
                )
            }
        }
        return true
    }

    internal suspend fun showErrorIfFailure(bitmapTransformation: BitmapTransformation): Boolean {
        val bitmapTransformationError = when (bitmapTransformation) {
            is BitmapTransformation.BitmapDecodingError ->
                MediaVerificationError.BITMAP_DECODING_ERROR
            is BitmapTransformation.SmallImageError ->
                MediaVerificationError.DOCUMENT_SMALL_IMAGE_SIZE
            //No Failure
            else -> return false
        }
        withContext(Dispatchers.Main) {
            bitmapTransformationError.run {
                navigation.navigateTo(
                    BaseErrorFragment.destination(prepareMediaErrorScreenData(title = getString(title), primaryCTALabel = getString(R.string.label_try_again)))
                )
            }
        }
        return true
    }


    companion object {
        internal const val ARG_NEW_STEP_STARTED = "ARG_NEW_STEP_STARTED"
    }
}