package com.getmati.mati_sdk.ui.data_prefetch

import androidx.lifecycle.SavedStateHandle
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.Metadata
import com.getmati.mati_sdk.managers.prefetch.ARG_DATA_PREFETCH
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataHolder
import com.getmati.mati_sdk.managers.prefetch.PrefetchDataSaveStateHandler
import com.getmati.mati_sdk.models.clean.SocketEvent
import com.getmati.mati_sdk.server.SocketManager
import com.getmati.mati_sdk.ui.data_prefetch.model.PrefetchedData
import com.getmati.mati_sdk.ui.utils.LocaleManager
import kotlinx.coroutines.ObsoleteCoroutinesApi
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import org.koin.core.component.KoinComponent

internal class DataPrefetchVm(private val savedState: SavedStateHandle,
                              private val socketManager: SocketManager,
                              private val dataPrefetchRepo: DataPrefetchRepo,
                              private val prefetchDataSaveStateHandler: PrefetchDataSaveStateHandler,
                              private val prefetchDataHolder: PrefetchDataHolder) : ViewModel(), KoinComponent {

    private val metadata by lazy { savedState.get<Metadata>(ARG_METADATA) }
    private val clientId by lazy { savedState.get<String>(ARG_CLIENT_ID) }
    private val flowId by lazy { savedState.get<String>(ARG_FLOW_ID) }
    private val config = metadata?.getConfig()

    //Flow Data State
    private val _state = MutableStateFlow<State>(State.Loading)
    val state: StateFlow<State> get() = _state

    //Other managers and supplementary classes
    @ObsoleteCoroutinesApi
    private val flowLoadingContext = newSingleThreadContext("RequestHandler")

    init {
        prefetchDataHolder.holdConfig(config)
        config?.fixedLanguage?.let { LocaleManager.persistLanguage(it) }

        if (savedState.contains(ARG_DATA_PREFETCH)) {
            _state.value = State.VerificationFlowLoading
        } else {
            loadFlowData()
        }
    }

    /**
     * Public function for retrial functionality
     */
    fun loadFlowData() {
        viewModelScope.launch(flowLoadingContext) {
            if(_state.value !is State.Loading ) {
                _state.value = State.Loading
            }
            val createAuthTokenState = dataPrefetchRepo.createAuthTokenIfNeeded(clientId, flowId, metadata, config)
            _state.value = createAuthTokenState
            if (createAuthTokenState is State.VerificationCreateSuccess) {
                prefetchDataHolder.holdPrefetchData(createAuthTokenState.prefetchedData)
                _state.value = State.VerificationFlowLoading
                loadVerificationFlow()
            }
        }
    }

    private suspend fun loadVerificationFlow() {
        prefetchDataSaveStateHandler.putPrefetchData(prefetchDataHolder, savedState)
        socketManager.connect()
        socketManager.eventFlow
            .filter { it is SocketEvent.RoomJoined }
            .map { it as SocketEvent.RoomJoined }
            .collect {
                prefetchDataHolder.holdVerificationFlow(it.verificationFlow)
                prefetchDataHolder.holdInitialInputs(it.inputs)
                prefetchDataHolder.holdWebContainerConfigs(it.webContainerConfig)
                prefetchDataSaveStateHandler.putAllData(prefetchDataHolder, savedState)
                _state.value = State.Success
            }
    }

    override fun onCleared() {
        if (_state.value !is State.Success) {
            socketManager.disconnect()
            prefetchDataHolder.clearPrefetchData()
        }
        super.onCleared()
    }

    internal sealed class State {
        object Loading : State()
        class VerificationCreateSuccess(val prefetchedData: PrefetchedData) : State()
        object VerificationFlowLoading : State()
        object Success : State()
        data class ErrorState(val errorCode: Int) : State()
    }
}


