package com.getmati.mati_sdk.ui.doc_hint

import android.graphics.Bitmap
import android.os.Bundle
import android.view.View
import android.widget.Toast
import androidx.activity.OnBackPressedCallback
import androidx.activity.result.ActivityResultLauncher
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.Constants.MIME_TYPE_IMAGE
import com.getmati.mati_sdk.Constants.MIME_TYPE_PDF
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.DocumentUploadEvent
import com.getmati.mati_sdk.analytics.events.Started
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentDocumentHintBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.*
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.NationalId
import com.getmati.mati_sdk.models.clean.ProofOfResidency
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.ExitFragment
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.document.DocumentCameraFragment
import com.getmati.mati_sdk.ui.document.DocumentPreviewFragment
import com.getmati.mati_sdk.ui.document.SelectPorTypeFragment
import com.getmati.mati_sdk.ui.document.getDocTitle
import com.getmati.mati_sdk.ui.utils.*
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.koin.androidx.viewmodel.ext.android.viewModel
import java.io.FileOutputStream

internal class DocumentHintFragment : KYCBaseFragment(R.layout.fragment_document_hint),
    ImagePicker {
    override val screenName = "documentHint"

    private val docPage: DocPage<*> by lazy { requireArguments().getParcelable(ARG_DOC_PAGE)!! }
    private val group by lazy { requireArguments().getInt(ARG_DOCUMENT_GROUP) }
    private val skippable by lazy { requireArguments().getBoolean(ARG_SKIPPABLE) }
    private val binding by viewBinding(FragmentDocumentHintBinding::bind)

    private val docSkipVm: DocSkipVm by viewModel()

    override fun configureToolbar(toolbar: MatiToolbar) {
        super.configureToolbar(toolbar)
        toolbar.setBackImageVisibile(docPage.isFrontSide)
    }

    private val getContentContract = GetContentForMimeTypes()

    override val itemPickResultLauncher: ActivityResultLauncher<String> = registerForActivityResult(
        getContentContract
    ) { uri ->
        uri?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.IO) {
                copyContentToCache(it)?.let { file ->
                    val filePath = file.path

                    val pdfToImageTransformation = requireContext().tryConvertPdfToImage(file)
                    if (!showErrorIfFailure(pdfToImageTransformation)) {
                        val previewPath = when (pdfToImageTransformation) {
                            is PdfToImageTransformation.Success -> pdfToImageTransformation.pathToPdf
                            else -> filePath
                        }

                        val rotationResult = rotateAndScaleDown(previewPath)
                        if (!showErrorIfFailure(rotationResult)) {
                            val scaledImagePath =
                                requireContext().getImagePath(docPage.document.id)

                            (rotationResult as BitmapTransformation.Success).bitmap.compress(
                                Bitmap.CompressFormat.JPEG,
                                100,
                                FileOutputStream(scaledImagePath)
                            )

                            withContext(Dispatchers.Main) {
                                docPage.document.run {
                                    if (this !is CustomDoc || !singleFile) {
                                        navigation.navigateTo(
                                            DocumentPreviewFragment.destination(
                                                DocPageStep(
                                                    docPage,
                                                    group,
                                                    previewPath,
                                                    scaledImagePath
                                                )
                                            )
                                        )
                                    } else {
                                        if (file.extension == SINGLE_FILE_VALID_EXTENSION) {
                                            navigation.navigateTo(
                                                DocumentPreviewFragment.destination(
                                                    DocPageStep(
                                                        docPage,
                                                        group,
                                                        previewPath,
                                                        filePath
                                                    )
                                                )
                                            )
                                        } else {
                                            Toast.makeText(
                                                requireContext(),
                                                getString(R.string.error_message_wrong_multi_custom_doc_file_format),
                                                Toast.LENGTH_LONG
                                            ).show()
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        if (savedInstanceState == null) {
            track(DocumentUploadEvent(docPage, Started(), group))
        }
        docPage.document.apply {
            if (this is CustomDoc && singleFile) {
                getContentContract.mimeTypes = arrayOf(MIME_TYPE_PDF)
            } else {
                getContentContract.mimeTypes = arrayOf(MIME_TYPE_IMAGE, MIME_TYPE_PDF)
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setListeners()
        setObserver()
    }

    private fun setObserver() {
        viewLifecycleOwner.lifecycleScope.launch {
            docSkipVm.state.collect {
                when (it) {
                    is DocSkipVm.State.Initial -> {
                        setValues()
                    }
                    is DocSkipVm.State.InProgress -> {
                        showProgressBar()
                    }
                    is DocSkipVm.State.SkipSuccess -> {
                        navigation.openNextStep()
                    }
                    is DocSkipVm.State.Error -> {
                        Toast.makeText(
                            requireContext(),
                            getString(R.string.label_something_went_wrong),
                            Toast.LENGTH_LONG
                        ).show()
                        docSkipVm.dropState()
                    }
                }
            }
        }
    }

    private fun showProgressBar() {
        val doc = docPage.document
        binding.actionPrimary.visibility =
            if (doc is CustomDoc && doc.singleFile) View.GONE else View.INVISIBLE
        binding.actionSecondaryImage.visibility =
            if (verificationFlow.denyUploadsFromMobileGallery) View.GONE
            else View.INVISIBLE
        binding.skipTv.visibility = if (skippable) View.INVISIBLE else View.GONE
        binding.progressBar.visibility = View.VISIBLE
    }

    private fun setValues() {
        binding.actionPrimary.visibility = View.VISIBLE
        binding.progressBar.visibility = View.GONE
        binding.actionPrimaryText.text = requireContext().getString(R.string.label_take_photo)

        if (verificationFlow.denyUploadsFromMobileGallery) {
            binding.actionSecondary.visibility = View.GONE
        } else {
            binding.actionSecondary.visibility = View.VISIBLE
            binding.actionSecondary.setSingleClickListener {
                track(UserAction("pickFromGalleryButton", Clicked(), screenName))
                pickImage()
            }
        }
        val doc = docPage.document
        val countryString = doc.country?.let {
            if (doc.region.isNullOrBlank()) {
                String.format("%s", doc.country?.name)
            } else {
                String.format("%s, %s", doc.country?.name, doc.region)
            }
        } ?: ""
        binding.innerTitleTvDocumentHintFragCountry.text = countryString
        binding.innerTitleTvDocumentHintFrag.text = getDocTitle(doc)

        if (doc is CustomDoc && doc.description.isNotEmpty()) {
            binding.descriptionTvDocHintFrag.visibility = View.VISIBLE
            binding.descriptionTvDocHintFrag.text = doc.description
        }

        if (doc.bothSidesNeeded) {
            val docSideString =
                requireContext().getString(if (docPage.isFrontSide) R.string.label_front_side else R.string.label_back_side)
            binding.innerTitleTvDocumentHintFragDocside.text = docSideString
        }
        when (doc) {
            is Passport -> {
                binding.actionSecondaryText.text =
                    requireContext().getString(R.string.label_attach_file)
                binding.actionSecondaryImage.loadImageResource(R.drawable.ic_attachfileicon)
                binding.mainIvDocumentHintFrag.loadImageResource(R.drawable.ic_passport)
            }
            is NationalId -> {
                binding.actionSecondaryImage.loadImageResource(R.drawable.ic_gallery)
                binding.actionSecondaryText.text =
                    requireContext().getString(R.string.label_gallery)
                val drawable = if (docPage.isFrontSide) R.drawable.ic_national_id_front
                    else R.drawable.ic_national_id_back
                binding.mainIvDocumentHintFrag.loadImageResource(drawable)
            }
            is DrivingLicense -> {
                binding.actionSecondaryText.text =
                    requireContext().getString(R.string.label_gallery)
                binding.actionSecondaryImage.loadImageResource(R.drawable.ic_gallery)
                val drawable = if (docPage.isFrontSide) R.drawable.ic_national_id_front
                else R.drawable.ic_driving_license_back
                binding.mainIvDocumentHintFrag.loadImageResource(drawable)
            }
            is CustomDoc -> {
                if (doc.singleFile) binding.actionPrimary.visibility = View.GONE
                binding.actionSecondaryText.text =
                    requireContext().getString(R.string.label_gallery)
                binding.actionSecondaryImage.loadImageResource(R.drawable.ic_gallery)
                binding.photoProgressBar.visibility = View.VISIBLE
                binding.mainIvDocumentHintFrag.loadRemoteImage(doc.hintImageUrls[if (docPage.isFrontSide) 0 else 1]) {
                    binding.photoProgressBar.visibility = View.INVISIBLE
                }
            }
            else -> throw IllegalArgumentException("Document type not found")
        }

        binding.skipTv.visibility = if (skippable) View.VISIBLE else View.GONE

    }

    private fun setListeners() {
        binding.actionPrimary.setSingleClickListener {
            track(UserAction("primaryButton", Clicked(), screenName))
            navigation.navigateTo(DocumentCameraFragment.destination(docPage, group))
        }
        binding.skipTv.setSingleClickListener {
            docSkipVm.skip(docPage.document.id, group)
        }
        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    if (!docPage.isFrontSide) {
                        navigation.navigateTo(ExitFragment.destination())
                    } else {
                        blockBackToPreviousStep.handleOnBackPressed()
                    }
                }
            })
    }

    companion object {
        @JvmStatic
        fun <T : Document> destination(
            docPage: DocPage<T>,
            documentGroup: Int,
            skippable: Boolean = false
        ): MatiDestination {
            return if (docPage.document is ProofOfResidency) {
                SelectPorTypeFragment.destination(
                    docPage as DocPage<ProofOfResidency>,
                    documentGroup
                )
            } else {
                MatiDestination(R.id.to_documentHint, Bundle().apply {
                    putParcelable(ARG_DOC_PAGE, docPage)
                    putInt(ARG_DOCUMENT_GROUP, documentGroup)
                    putBoolean(ARG_SKIPPABLE, skippable)
                })
            }
        }

        private const val SINGLE_FILE_VALID_EXTENSION = "pdf"
        private const val ARG_DOC_PAGE = "ARG_DOC_PAGE"
        private const val ARG_DOCUMENT_GROUP = "ARG_DOCUMENT_GROUP"
        private const val ARG_SKIPPABLE = "ARG_SKIPPABLE"
    }
}