package com.getmati.mati_sdk.ui.document

import com.getmati.mati_sdk.models.clean.*
import com.getmati.mati_sdk.models.clean.input.Input

internal class CountriesRepo {

    private val defaultPriorityCountries = listOf("US", "AR", "MX", "BR", "CL", "CO", "PE", "VE")
    private val all: List<Country>
        get() = default

    val defaultFirstCountry: Country? = getCountryByCode(defaultPriorityCountries.first())

    fun filterSupported(supportedCountryCodes: List<String>? = null): List<Country> {
        return when {
            supportedCountryCodes.isNullOrEmpty() -> all
            else -> all.filter { it.code in supportedCountryCodes }
        }
    }

    fun shouldShowSelectRegion(selectedCountryCode: String, doc: Document): Boolean {
        return selectedCountryCode in listOf(Countries.US.name, Countries.CA.name)
                && (doc is DrivingLicense || doc is NationalId || doc is CustomDoc)
    }

    fun getCountryByCode(code: String?): Country? {
        return all.firstOrNull { it.code == code }
    }

    fun reorderWithAllowedRegions(
        source: List<Country> = all,
        currentCountryCode: String?,
        allowedRegions: List<String>
    ): List<Country> {
        if (currentCountryCode == null) {
            val priorityCountries = defaultPriorityCountries
                    .filter { defaultCountryCode -> source.any { it.code == defaultCountryCode } }
                    .mapNotNull { getCountryByCode(it) }
            val filterDefaults = source
                .filter { !defaultPriorityCountries.contains(it.code) }
                .toMutableList()
            filterDefaults.addAll(0, priorityCountries)
            return filterDefaults
        }
        val allCountries = source
            .filter { !allowedRegions.contains(it.code) }
            .filter { it.code != currentCountryCode }
            .toMutableList()

        val allowedCountries = allowedRegions
            .filter { defaultCountryCode -> source.any { it.code == defaultCountryCode } }
            .mapNotNull { getCountryByCode(it) }
        allCountries.addAll(0, allowedCountries)
        getCountryByCode(currentCountryCode)?.let {
            if (source.contains(it)) {
                allCountries.add(0, it)
            }
        }
        return allCountries
    }

    fun getCountryFromInput(inputs: List<Input>): Country? {
        val countryCode =
            inputs.firstOrNull { it.inputData?.countryCode != null }?.inputData?.countryCode
                ?: return null
        return getCountryByCode(countryCode)
    }

    companion object {
        val default = listOf(
            Country("AD", "Andorra", 376, true),
            Country("AE", "United Arab Emirates", 971, true),
            Country("AF", "Afghanistan", 93, true),
            Country("AG", "Antigua & Barbuda", 1268, true),
            Country("AI", "Anguilla", 1264, true),
            Country("AL", "Albania", 355, true),
            Country("AM", "Armenia", 374, true),
            Country("AO", "Angola", 244, true),
            Country("AR", "Argentina", 54, true),
            Country("AS", "American Samoa", 1684, true),
            Country("AT", "Austria", 43, true),
            Country("AU", "Australia", 61, true),
            Country("AW", "Aruba", 297, true),
            Country("AZ", "Azerbaijan", 994, true),
            Country("BA", "Bosnia & Herzegovina", 387, true),
            Country("BB", "Barbados", 1246, true),
            Country("BD", "Bangladesh", 880, true),
            Country("BE", "Belgium", 32, true),
            Country("BF", "Burkina Faso", 226, true),
            Country("BG", "Bulgaria", 359, true),
            Country("BH", "Bahrain", 973, true),
            Country("BI", "Burundi", 257, true),
            Country("BJ", "Benin", 229, true),
            Country("BM", "Bermuda", 1441, true),
            Country("BN", "Brunei", 673, true),
            Country("BO", "Bolivia", 591, true),
            Country("BR", "Brazil", 55, true),
            Country("BS", "Bahamas", 1242, true),
            Country("BT", "Bhutan", 975, true),
            Country("BW", "Botswana", 267, true),
            Country("BY", "Belarus", 375, false),
            Country("BZ", "Belize", 501, true),
            Country("CA", "Canada", 1, true),
            Country("CD", "Congo - Kinshasa", 243, true),
            Country("CF", "Central African Republic", 236, true),
            Country("CG", "Congo - Brazzaville", 242, true),
            Country("CH", "Switzerland", 41, true),
            Country("CI", "Côte d’Ivoire", 225, true),
            Country("CL", "Chile", 56, true),
            Country("CM", "Cameroon", 237, true),
            Country("CN", "China", 86, true),
            Country("CO", "Colombia", 57, true),
            Country("CR", "Costa Rica", 506, true),
            Country("CU", "Cuba", 53, false),
            Country("CV", "Cape Verde", 238, true),
            Country("CY", "Cyprus", 357, true),
            Country("CW", "Curaçao", 599, true),
            Country("CZ", "Czechia", 420, true),
            Country("DE", "Germany", 49, true),
            Country("DJ", "Djibouti", 253, true),
            Country("DK", "Denmark", 45, true),
            Country("DM", "Dominica", 1767, true),
            Country("DO", "Dominican Republic", 1849, true),
            Country("DZ", "Algeria", 213, true),
            Country("EC", "Ecuador", 593, true),
            Country("EE", "Estonia", 372, true),
            Country("EG", "Egypt", 20, true),
            Country("ER", "Eritrea", 291, true),
            Country("ES", "Spain", 34, true),
            Country("ET", "Ethiopia", 251, true),
            Country("FI", "Finland", 358, true),
            Country("FJ", "Fiji", 679, true),
            Country("FR", "France", 33, true),
            Country("GA", "Gabon", 241, true),
            Country("GB", "United Kingdom", 44, true),
            Country("GD", "Grenada", 1473, true),
            Country("GE", "Georgia", 995, true),
            Country("GH", "Ghana", 233, true),
            Country("GM", "Gambia", 220, true),
            Country("GN", "Guinea", 224, true),
            Country("GQ", "Equatorial Guinea", 240, true),
            Country("GR", "Greece", 30, true),
            Country("GT", "Guatemala", 502, true),
            Country("GU", "Guam", 1671, true),
            Country("GW", "Guinea-Bissau", 245, true),
            Country("GY", "Guyana", 592, true),
            Country("HK", "Hong Kong SAR China", 852, true),
            Country("HN", "Honduras", 504, true),
            Country("HR", "Croatia", 385, true),
            Country("HT", "Haiti", 509, true),
            Country("HU", "Hungary", 36, true),
            Country("ID", "Indonesia", 62, true),
            Country("IE", "Ireland", 353, true),
            Country("IL", "Israel", 972, true),
            Country("IN", "India", 91, true),
            Country("IQ", "Iraq", 964, false),
            Country("IR", "Iran", 98, false),
            Country("IS", "Iceland", 354, true),
            Country("IT", "Italy", 39, true),
            Country("JM", "Jamaica", 1876, true),
            Country("JO", "Jordan", 962, true),
            Country("JP", "Japan", 81, true),
            Country("KE", "Kenya", 254, true),
            Country("KG", "Kyrgyzstan", 996, true),
            Country("KH", "Cambodia", 855, true),
            Country("KI", "Kiribati", 686, true),
            Country("KM", "Comoros", 269, true),
            Country("KN", "St. Kitts & Nevis", 1869, true),
            Country("KP", "North Korea", 850, false),
            Country("KR", "South Korea", 82, true),
            Country("XK", "Kosovo", 383, true),
            Country("KW", "Kuwait", 965, true),
            Country("KY", "Cayman Islands", 1345, true),
            Country("KZ", "Kazakhstan", 7, true),
            Country("LA", "Laos", 856, true),
            Country("LB", "Lebanon", 961, true),
            Country("LC", "St. Lucia", 1758, true),
            Country("LI", "Liechtenstein", 423, true),
            Country("LK", "Sri Lanka", 238, true),
            Country("LR", "Liberia", 231, false),
            Country("LS", "Lesotho", 266, true),
            Country("LT", "Lithuania", 370, true),
            Country("LU", "Luxembourg", 352, true),
            Country("LV", "Latvia", 371, true),
            Country("LY", "Libya", 218, true),
            Country("MA", "Morocco", 212, true),
            Country("MC", "Monaco", 377, true),
            Country("MD", "Moldova", 373, true),
            Country("ME", "Montenegro", 382, true),
            Country("MF", "Saint Marteen", 590, true),
            Country("MG", "Madagascar", 261, true),
            Country("MK", "Macedonia", 389, true),
            Country("ML", "Mali", 223, true),
            Country("MM", "Myanmar (Burma)", 95, true),
            Country("MN", "Mongolia", 976, true),
            Country("MO", "Macau", 853, true),
            Country("MP", "Northern Mariana Islands", 1670, true),
            Country("MR", "Mauritania", 222, true),
            Country("MT", "Malta", 356, true),
            Country("MU", "Mauritius", 230, true),
            Country("MV", "Maldives", 960, true),
            Country("MW", "Malawi", 265, true),
            Country("MX", "Mexico", 52, true),
            Country("MY", "Malaysia", 60, true),
            Country("MZ", "Mozambique", 258, true),
            Country("NA", "Namibia", 264, true),
            Country("NE", "Niger", 227, true),
            Country("NG", "Nigeria", 234, true),
            Country("NI", "Nicaragua", 505, true),
            Country("NL", "Netherlands", 31, true),
            Country("NO", "Norway", 47, true),
            Country("NP", "Nepal", 977, true),
            Country("NR", "Nauru", 674, true),
            Country("NU", "Niue", 683, true),
            Country("NZ", "New Zealand", 64, true),
            Country("OM", "Oman", 968, true),
            Country("PA", "Panama", 507, true),
            Country("PE", "Peru", 51, true),
            Country("PG", "Papua New Guinea", 675, true),
            Country("PH", "Philippines", 63, true),
            Country("PK", "Pakistan", 92, true),
            Country("PL", "Poland", 48, true),
            Country("PR", "Puerto Rico", 1939, true),
            Country("PS", "Palestinian Territories", 970, true),
            Country("PT", "Portugal", 351, true),
            Country("PY", "Paraguay", 595, true),
            Country("QA", "Qatar", 974, true),
            Country("RO", "Romania", 40, true),
            Country("RS", "Serbia", 381, true),
            Country("RU", "Russia", 7, true),
            Country("RW", "Rwanda", 250, true),
            Country("SA", "Saudi Arabia", 966, true),
            Country("SB", "Solomon Islands", 677, true),
            Country("SC", "Seychelles", 248, true),
            Country("SD", "Sudan", 249, false),
            Country("SE", "Sweden", 46, true),
            Country("SG", "Singapore", 65, true),
            Country("SI", "Slovenia", 386, true),
            Country("SK", "Slovakia", 421, true),
            Country("SL", "Sierra Leone", 232, true),
            Country("SM", "San Marino", 378, true),
            Country("SN", "Senegal", 221, true),
            Country("SO", "Somalia", 252, true),
            Country("SR", "Suriname", 597, true),
            Country("SS", "South Sudan", 211, true),
            Country("ST", "São Tomé & Príncipe", 239, true),
            Country("SV", "El Salvador", 503, true),
            Country("SY", "Syria", 963, false),
            Country("SZ", "Swaziland", 268, true),
            Country("TC", "Turks and Caicos", 1649, true),
            Country("TD", "Chad", 235, true),
            Country("TG", "Togo", 228, true),
            Country("TH", "Thailand", 66, true),
            Country("TJ", "Tajikistan", 992, true),
            Country("TL", "Timor-Leste", 670, true),
            Country("TM", "Turkmenistan", 993, true),
            Country("TN", "Tunisia", 216, true),
            Country("TO", "Tonga", 676, true),
            Country("TR", "Turkey", 90, true),
            Country("TT", "Trinidad & Tobago", 1868, true),
            Country("TV", "Tuvalu", 688, true),
            Country("TZ", "Tanzania", 255, true),
            Country("TW", "Taiwan", 886, true),
            Country("UA", "Ukraine", 380, true),
            Country("UG", "Uganda", 256, true),
            Country("US", "United States", 1, true),
            Country("UY", "Uruguay", 598, true),
            Country("UZ", "Uzbekistan", 998, true),
            Country("VA", "Vatican City", 379, true),
            Country("VC", "Saint Vincent and the Grenadines", 1784, true),
            Country("VE", "Venezuela", 58, true),
            Country("VI", "Virgin Islands (U.S.)", 1340, true),
            Country("VN", "Vietnam", 84, true),
            Country("VU", "Vanuatu", 678, true),
            Country("WS", "Samoa", 685, true),
            Country("YE", "Yemen", 967, true),
            Country("YT", "Mayotte", 262, true),
            Country("ZA", "South Africa", 27, true),
            Country("ZM", "Zambia", 260, true),
            Country("ZW", "Zimbabwe", 263, false)
        )
    }
}