package com.getmati.mati_sdk.ui.document

import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.view.View
import android.widget.ImageView
import androidx.appcompat.app.AlertDialog
import androidx.lifecycle.lifecycleScope
import coil.load
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentCpfEntryBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.DocMetadata
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.NationalId
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.utils.loadImageFromFileUrl
import com.getmati.mati_sdk.ui.utils.loadImageResource
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.CpfEditText
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch


internal class CpfEntryFragment : KYCBaseFragment(R.layout.fragment_cpf_entry) {

    override val screenName = "cpfEntryFragment"

    private val docPageStep: DocPageStep<NationalId> by lazy {
        requireArguments().getParcelable(ARG_DOC_PAGE_STEP)!!
    }
    private val showFrames by lazy { requireArguments().getBoolean(ARG_SHOW_FRAMES) }


    private val binding by viewBinding(FragmentCpfEntryBinding::bind)

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setupUI()
        setUpListeners()
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        super.configureToolbar(toolbar)
        toolbar.setBackImageVisibile(true)
    }

    private fun setupUI() {
        val doc = docPageStep.docPage.document
        val countryString = doc.country?.let {
            if (doc.region.isNullOrBlank()) {
                String.format("%s", doc.country?.name)
            } else {
                String.format("%s, %s", doc.country?.name, doc.region)
            }
        } ?: ""

        binding.tvTitle.text = getDocTitle(doc)
        binding.tvTitleCountry.text = countryString
        binding.ivCard.loadImageFromFileUrl(docPageStep.previewPath)
    }

    private fun setUpListeners() {
        viewLifecycleOwner.lifecycleScope.launch {
            binding.etCpf.state.collect {
                when (it) {
                    is CpfEditText.State.Filled -> {
                        binding.btnSkipSubmit.text = getString(R.string.cpf_button_submit)
                        binding.btnSkipSubmit.isEnabled = true
                        binding.btnSkipSubmit.alpha = ALPHA_ENABLED_ACTION_BUTTON
                        binding.btnSkipSubmit.setSingleClickListener {
                            val result = docPageStep.run {
                                copy(
                                    docPage = docPage.copy(
                                        document = docPage.document.copy(
                                            metadata = DocMetadata(taxpayerNumber = binding.etCpf.text)
                                        )
                                    )
                                )
                            }
                            navigation.navigateTo(
                                DocumentUploadFragment.destination(
                                    result, showFrames
                                )
                            )
                        }
                    }
                    is CpfEditText.State.Empty -> {
                        binding.btnSkipSubmit.text = getString(R.string.cpf_button_skip_submit)
                        binding.btnSkipSubmit.isEnabled = true
                        binding.btnSkipSubmit.alpha = ALPHA_ENABLED_ACTION_BUTTON
                        binding.btnSkipSubmit.setSingleClickListener {
                            showAlertDialog()
                        }
                    }
                    is CpfEditText.State.NotComplete -> {
                        binding.btnSkipSubmit.text = getString(R.string.cpf_button_submit)
                        binding.btnSkipSubmit.isEnabled = false
                        binding.btnSkipSubmit.alpha = ALPHA_DISABLED_ACTION_BUTTON
                    }
                }
            }
        }
    }

    private fun showAlertDialog() {
        val dialogBuilder = AlertDialog.Builder(requireContext())
        val dialogView = layoutInflater.inflate(R.layout.no_cpf_alert_dialog, null)
        dialogBuilder.setView(dialogView)
        val dialog = dialogBuilder.create()
        dialogView.findViewById<View>(R.id.btnSkipSubmit).setOnClickListener {
            dialog.cancel()
            navigation.navigateTo(DocumentUploadFragment.destination(docPageStep, showFrames))
        }
        dialogView.findViewById<ImageView>(R.id.ivClose).setOnClickListener {
            dialog.cancel()
        }
        dialog.window?.setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        dialog.show()
    }

    companion object {
        private const val ARG_DOC_PAGE_STEP = "ARG_DOC_PAGE"
        private const val ARG_SHOW_FRAMES = "ARG_SHOW_FRAMES"

        private const val ALPHA_DISABLED_ACTION_BUTTON = 0.5f
        private const val ALPHA_ENABLED_ACTION_BUTTON = 1f


        fun destination(docPageStep: DocPageStep<*>, showFrames: Boolean = false) =
            MatiDestination(
                R.id.to_cpfEntry,
                Bundle().apply {
                    putParcelable(ARG_DOC_PAGE_STEP, docPageStep)
                    putBoolean(ARG_SHOW_FRAMES, showFrames)
                }
            )
    }
}