package com.getmati.mati_sdk.ui.document

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.server.verification.VerificationError
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch

internal class DocUploadVm(private val docUploadRepository: DocUploadRepository) : ViewModel() {

    private val _state = MutableStateFlow<State>(State.Loading)
    val state: StateFlow<State> get() = _state

    fun startUpload(docPageStep: DocPageStep<*>) {
        docPageStep.run {
            if (docPage.document is CustomDoc) {
                uploadCustomDocument(this as DocPageStep<CustomDoc>)
            } else {
                uploadDocument(this)
            }
        }
    }

    private fun uploadDocument(docPageStep: DocPageStep<*>) {
        _state.value = State.Loading
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = docUploadRepository.uploadDoc(docPageStep).run {
                if (error == null || error.type == MediaVerificationError.INPUT_LOCKED) {
                    State.Success
                } else {
                    State.Error(error)
                }
            }
        }
    }

    private fun uploadCustomDocument(docPageStep: DocPageStep<CustomDoc>) {
        _state.value = State.Loading
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = docUploadRepository.uploadCustomDoc(docPageStep).run {
                if (error == null || error.type == MediaVerificationError.INPUT_LOCKED) {
                    State.Success
                } else {
                    State.Error(error)
                }
            }
        }
    }

    sealed class State {
        object Loading : State()
        object Success : State()
        data class Error(val error: VerificationError) : State()
    }
}