package com.getmati.mati_sdk.ui.document

import android.graphics.Bitmap
import android.net.Uri
import android.os.Bundle
import android.view.View
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.content.res.AppCompatResources
import androidx.camera.core.CameraSelector
import androidx.camera.core.ImageCaptureException
import androidx.camera.view.PreviewView
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.Constants.MIME_TYPE_IMAGE
import com.getmati.mati_sdk.Constants.MIME_TYPE_PDF
import com.getmati.mati_sdk.Constants.TEMP_FILE_NAME
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentDocumentCameraBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.*
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.models.clean.NationalId
import com.getmati.mati_sdk.models.clean.ProofOfResidency
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.camera.PhotoCameraFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.permission_denial.PermissionDenialInfoFragment
import com.getmati.mati_sdk.ui.utils.*
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.File
import java.io.FileOutputStream

internal class DocumentCameraFragment : PhotoCameraFragment(R.layout.fragment_document_camera),
    ImagePicker, PhotoCapture {

    override val screenName = "documentCamera"

    private val docPage: DocPage<*> by lazy { requireArguments().getParcelable(ARG_DOC_PAGE)!! }
    private val group by lazy { requireArguments().getInt(ARG_DOC_GROUP) }
    private val binding by viewBinding(FragmentDocumentCameraBinding::bind)

    override var lensFacing = CameraSelector.LENS_FACING_BACK
    override val flipHorizontally get() = false

    override var latestTmpUri: Uri? = null

    override val viewFinder: PreviewView get() = binding.previewView

    override val takePhotoResultLauncher = registerForActivityResult(ActivityResultContracts.TakePicture()) { isSuccess ->
        if (isSuccess) {
            latestTmpUri?.let { _ ->
                when (val rotationResult = rotateAndScaleDown(requireContext().getImagePath(TEMP_FILE_NAME))) {
                    is BitmapTransformation.Success -> {
                        val dstPath = requireContext().getImagePath(docPage.document.id)
                        rotationResult.bitmap.compress(Bitmap.CompressFormat.JPEG, 100, FileOutputStream(dstPath))
                        navigation.navigateTo(DocumentPreviewFragment.destination(DocPageStep(docPage, group, dstPath, dstPath)))
                    }
                    else -> {
                        MediaVerificationError.OTHER.run {
                            navigation.navigateTo(
                                BaseErrorFragment.destination(
                                    prepareMediaErrorScreenData(
                                        title = getString(title),
                                        primaryCTALabel = getString(R.string.label_try_again)
                                    )
                                )
                            )
                        }
                    }
                }
            }
        }
    }

    override val itemPickResultLauncher: ActivityResultLauncher<String> = registerForActivityResult(
        GetContentForMimeTypes().apply { mimeTypes = arrayOf(MIME_TYPE_IMAGE, MIME_TYPE_PDF) }
    ) { uri ->
        uri?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.IO) {
                copyContentToCache(it)?.let { file ->
                    val pdfToImageTransformation = requireContext().tryConvertPdfToImage(file)
                    if (!showErrorIfFailure(pdfToImageTransformation)) {
                        val previewPath = when(pdfToImageTransformation) {
                            is PdfToImageTransformation.Success -> pdfToImageTransformation.pathToPdf
                            else -> file.path
                        }

                        val rotationResult = rotateAndScaleDown(previewPath)
                        if (!showErrorIfFailure(rotationResult)) {
                            val scaledImagePath = requireContext().getImagePath(docPage.document.id)
                            (rotationResult as BitmapTransformation.Success).bitmap.compress(
                                Bitmap.CompressFormat.JPEG,
                                100,
                                FileOutputStream(scaledImagePath)
                            )

                            withContext(Dispatchers.Main) {
                                navigation.navigateTo(
                                    DocumentPreviewFragment.destination(
                                        DocPageStep(
                                            docPage,
                                            group,
                                            previewPath,
                                            scaledImagePath
                                        )
                                    )
                                )
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setValues()
    }

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    private fun setValues() {
        setUpCaptureButton()
        setUpInfoLabel()
        setUpOverlay()
        setUpGalleryIv()
    }

    private fun setUpCaptureButton() {
        binding.actionCapture.setOnClickListener {
            it.isEnabled = false
            track(UserAction("capturePhotoButton", Clicked(), screenName))
            takePhoto()
        }
    }

    private fun setUpInfoLabel() {
        docPage.document.apply {
            val documentLabel = when (docPage.document) {
                is DrivingLicense, is NationalId, is CustomDoc -> {
                    "${getDocTitle(docPage.document)}: ${
                        getString(
                            if (docPage.isFrontSide)
                                R.string.label_front_side
                            else
                                R.string.label_back_side
                        )
                    }\n"
                }
                else -> {
                    getDocTitle(docPage.document)
                }
            }
            val locationLabel =
                if (region.isNullOrBlank()) country?.name ?: "" else "${country!!.name}, $region"
            binding.labelInfoTakeDocPhotoFrag.text = "$documentLabel\n$locationLabel"
        }
    }

    private fun setUpOverlay() {
        binding.overlay.documentPreviewImage = AppCompatResources.getDrawable(
            requireContext(), when (docPage.document) {
                is DrivingLicense, is NationalId, is CustomDoc -> if (docPage.isFrontSide)
                    R.drawable.ic_dl_camera else R.drawable.ic_dl_camera_back
                is Passport -> R.drawable.ic_passport_camera
                is ProofOfResidency -> R.drawable.ic_por_camera
                else -> throw IllegalArgumentException("Unhandled document")
            }
        )
        viewLifecycleOwner.lifecycleScope.launch {
            delay(DOCUMENT_IMAGE_AUTO_HIDE_DELAY)
            binding.overlay.documentPreviewImage = null
        }
    }

    private fun setUpGalleryIv() {
        if (verificationFlow.denyUploadsFromMobileGallery) {
            binding.actionGallery.visibility = View.GONE
        } else {
            binding.actionGallery.visibility = View.VISIBLE
            binding.actionGallery.setSingleClickListener {
                track(UserAction("pickFromGalleryButton", Clicked(), screenName))
                pickImage()
            }
        }
    }

    override fun onImageSaved(file: File) {
        view?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.IO) {
                context?.let { context ->
                    when (val rotationResult = rotateAndScaleDown(file.absolutePath)) {
                        is BitmapTransformation.Success -> {
                            val scaledImagePath = context.getImagePath(docPage.document.id)
                            rotationResult.bitmap.compress(
                                Bitmap.CompressFormat.JPEG,
                                100,
                                FileOutputStream(scaledImagePath)
                            )
                            val pageStep =
                                DocPageStep(docPage, group, file.absolutePath, scaledImagePath)
                            withContext(Dispatchers.Main) {
                                navigation.navigateTo(
                                    DocumentPreviewFragment.destination(
                                        pageStep,
                                        true
                                    )
                                )
                            }
                        }
                        else -> {
                            withContext(Dispatchers.Main) {
                                MediaVerificationError.OTHER.run {
                                    navigation.navigateTo(
                                        BaseErrorFragment.destination(
                                            prepareMediaErrorScreenData(
                                                title = getString(title),
                                                subHeading = "",
                                                primaryCTALabel = getString(R.string.label_try_again)
                                            )
                                        )
                                    )
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onImageCaptureException(exc: ImageCaptureException) {
        view?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
                binding.actionCapture.isEnabled = true
            }
            takePhotoWithOtherApp()
        }
    }

    override fun onUseCaseBindingFailed(exc: Exception) {
        takePhotoWithOtherApp()
    }

    override fun onPermissionDenied(vararg permission: String) {
        onPermissionRationaleShown()
    }

    override fun onPermissionPermanentlyDenied(permission: String) {
        navigation.navigateTo(PermissionDenialInfoFragment.destination(permission))
    }

    override fun onFrontBackCamerasNotFound() {
        navigation.navigateTo(
            BaseErrorFragment.destination(
                prepareMediaErrorScreenData(
                    title = getString(R.string.mati_hardware_error_heading),
                    subHeading = getString(R.string.mati_hardware_error_desc),
                    primaryCTALabel = getString(R.string.label_try_again)
                )
            )
        )
    }

    companion object {
        private const val DOCUMENT_IMAGE_AUTO_HIDE_DELAY = 3000L

        private const val ARG_DOC_PAGE = "ARG_DOC_PAGE"
        private const val ARG_DOC_GROUP = "ARG_DOC_GROUP"

        fun destination(docPage: DocPage<*>, group: Int) = MatiDestination(
            R.id.to_documentCamera,
            Bundle().apply {
                putParcelable(ARG_DOC_PAGE, docPage)
                putInt(ARG_DOC_GROUP, group)
            })
    }
}