package com.getmati.mati_sdk.ui.document

import android.os.Bundle
import android.view.View
import android.widget.ImageView
import androidx.constraintlayout.widget.ConstraintSet
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.DocumentUploadEvent
import com.getmati.mati_sdk.analytics.events.Failed
import com.getmati.mati_sdk.analytics.events.Processed
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentDocumentUploadBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.doc_hint.DocumentHintFragment
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaProcessingErrorScreenData
import com.getmati.mati_sdk.ui.utils.Util
import com.getmati.mati_sdk.ui.utils.loadImageFromFileUrl
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.MatiToolbar
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class DocumentUploadFragment : KYCBaseFragment(R.layout.fragment_document_upload) {
    override val screenName = "documentUpload"

    private val binding by viewBinding(FragmentDocumentUploadBinding::bind)
    private val docPageStep: DocPageStep<*> by lazy { requireArguments().getParcelable(ARG_DOC_PAGE_STEP)!! }

    private val showFrames by lazy { requireArguments().getBoolean(ARG_SHOW_FRAMES) }
    private val retryCount get() = requireArguments().getInt(ARG_RETRY_COUNT, 0)

    private fun incrementRetryCount() = requireArguments().putInt(ARG_RETRY_COUNT, retryCount + 1)
    private val docUploadVm by viewModel<DocUploadVm>()

    override fun configureToolbar(toolbar: MatiToolbar) {
        toolbar.changeTheme(MatiToolbar.Theme.NONE)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        docUploadVm.startUpload(docPageStep)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setUpImageView()
        setUpObserver()
    }

    private fun setUpObserver() {
        viewLifecycleOwner.lifecycleScope.launch {
            docUploadVm.state.collect { documentUploadState ->
                when(documentUploadState){
                    is DocUploadVm.State.Loading ->{}
                    is DocUploadVm.State.Error ->{
                        val (iconId, error, errorDescription) =  MediaVerificationError.byId(documentUploadState.error.type.id).let { mediaVerificationError ->
                            Triple( mediaVerificationError.iconId,
                                requireContext().getString(mediaVerificationError.title),
                                requireContext().getString(mediaVerificationError.subtitle)
                            )
                        }
                        val destination = BaseErrorFragment.destination(
                            prepareMediaProcessingErrorScreenData(iconId,error,errorDescription,getString(R.string.label_try_again))
                        )
                        handleError(destination, documentUploadState.error.type.id)
                    }
                    is DocUploadVm.State.Success ->{
                        track(DocumentUploadEvent(docPageStep.docPage, Processed(), docPageStep.group))
                        docPageStep.docPage.run {
                            if (document.bothSidesNeeded && isFrontSide) {
                                val nextPage = docPageStep.docPage.copy(page = page + 1)
                                navigation.navigateTo(
                                    DocumentHintFragment.destination(nextPage, docPageStep.group)
                                )
                            } else {
                                navigation.openNextStep()
                            }
                        }
                    }
                }
            }
        }
    }

    private fun setUpImageView() {
        if (!showFrames) {
            binding.overlay.visibility = View.INVISIBLE
            val constraintSet = ConstraintSet()
            constraintSet.clone(binding.root)
            constraintSet.connect(
                R.id.main_iv,
                ConstraintSet.BOTTOM,
                R.id.title_tv_confirm,
                ConstraintSet.TOP,
                Util.dpToPx(requireContext(), BOTTOM_MARGIN_MAIN_IV_IN_DP)
            )
            constraintSet.applyTo(binding.root)
            binding.mainIv.scaleType = ImageView.ScaleType.FIT_CENTER
        }
        binding.mainIv.loadImageFromFileUrl(docPageStep.previewPath)
    }

    private fun handleError(destination: MatiDestination, code: String, message: String? = null) {
        track(DocumentUploadEvent(docPageStep.docPage, Failed(code, message, retryCount), docPageStep.group))
        incrementRetryCount()
        navigation.back()
        navigation.navigateTo(destination)
    }

    companion object {
        private const val ARG_DOC_PAGE_STEP = "ARG_DOC_PAGE_STEP"
        private const val ARG_SHOW_FRAMES = "ARG_SHOW_FRAMES"
        private const val ARG_RETRY_COUNT = "ARG_RETRY_COUNT"

        private const val BOTTOM_MARGIN_MAIN_IV_IN_DP =
            DocumentPreviewFragment.BOTTOM_MARGIN_MAIN_IV_IN_DP


        fun destination(docPageStep: DocPageStep<*>, showFrames: Boolean) =
            MatiDestination(
                R.id.to_documentUpload,
                Bundle().apply {
                    putParcelable(ARG_DOC_PAGE_STEP, docPageStep)
                    putBoolean(ARG_SHOW_FRAMES, showFrames)

                }
            )
    }
}