package com.getmati.mati_sdk.ui.document

import android.graphics.Bitmap
import android.graphics.Typeface
import android.os.Bundle
import android.text.Spannable
import android.text.SpannableString
import android.text.style.StyleSpan
import android.view.View
import androidx.activity.result.ActivityResultLauncher
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.Constants.MIME_TYPE_IMAGE
import com.getmati.mati_sdk.Constants.MIME_TYPE_PDF
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentPorHintBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.DocPage
import com.getmati.mati_sdk.models.clean.DocPageStep
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.models.clean.PorType
import com.getmati.mati_sdk.models.clean.ProofOfResidency
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.utils.*
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.FileOutputStream
import java.text.SimpleDateFormat
import java.util.*

internal class ProofOfResidenceHintFragment : KYCBaseFragment(R.layout.fragment_por_hint),
    ImagePicker {
    override val screenName = "proofOfResidencyHint"
    private val binding by viewBinding(FragmentPorHintBinding::bind)

    private val docPage: DocPage<ProofOfResidency> by lazy {
        requireArguments().getParcelable(
            ARG_DOC_PAGE
        )!!
    }
    private val group by lazy { requireArguments().getInt(ARG_DOCUMENT_GROUP) }

    override val itemPickResultLauncher: ActivityResultLauncher<String> = registerForActivityResult(
        GetContentForMimeTypes().apply { mimeTypes = arrayOf(MIME_TYPE_IMAGE, MIME_TYPE_PDF) }
    ) { uri ->
        uri?.let {
            viewLifecycleOwner.lifecycleScope.launch(Dispatchers.IO) {
                copyContentToCache(it)?.let { file ->
                    val pdfToImageTransformation = requireContext().tryConvertPdfToImage(file)
                    if (!showErrorIfFailure(pdfToImageTransformation)) {
                        val previewPath = when(pdfToImageTransformation) {
                            is PdfToImageTransformation.Success -> pdfToImageTransformation.pathToPdf
                            else -> file.path
                        }

                        val rotationResult = rotateAndScaleDown(previewPath)
                        if (!showErrorIfFailure(rotationResult)) {
                            val scaledImagePath = requireContext().getImagePath(docPage.document.id)
                            (rotationResult as BitmapTransformation.Success).bitmap.compress(
                                Bitmap.CompressFormat.JPEG,
                                100,
                                FileOutputStream(scaledImagePath)
                            )
                            val pageStep = DocPageStep(docPage, group, previewPath, scaledImagePath)
                            withContext(Dispatchers.Main) {
                                navigation.navigateTo(DocumentPreviewFragment.destination(pageStep))
                            }
                        }
                    }
                }
            }
        }
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        docPage.document.run {
            binding.titleTv.text = requireContext().getString(porType!!.titleRes)
            binding.subtitleTv.text = if (region.isNullOrBlank()) {
                String.format("%s", country!!.name)
            } else {
                String.format("%s, %s", country!!.name, region)
            }

            binding.mainIv.loadImageResource(R.drawable.ic_proof_noborder)

            binding.secondHintTv.text = when (porType) {
                PorType.UTILITY_BILL -> getString(R.string.select_por_second_hint_utility_bill)
                PorType.BANK_STATEMENT -> getString(R.string.select_por_second_hint_bank_statement)
            }
        }

        setUpValidityHint()

        binding.actionPrimaryText.text = requireContext().getString(R.string.label_take_photo)
        binding.actionPrimary.setSingleClickListener {
            track(UserAction("primaryButton", Clicked(), screenName))
            navigation.navigateTo(DocumentCameraFragment.destination(docPage, group))
        }
        setUpSecondaryAction()

        binding.actionSecondaryImage.loadImageResource(R.drawable.ic_attachfileicon)
        binding.actionSecondaryText.text = requireContext().getString(R.string.label_attach_file)
    }

    private fun setUpSecondaryAction() {
        val secondaryAction = binding.actionSecondary
        if (verificationFlow.denyUploadsFromMobileGallery) {
            secondaryAction.visibility = View.GONE
        } else {
            secondaryAction.visibility = View.VISIBLE
            secondaryAction.setSingleClickListener {
                track(UserAction("pickFromGalleryButton", Clicked(), screenName))
                pickImage()
            }
        }
    }

    private fun setUpValidityHint() {
        val validityLabel = getString(R.string.label_por_document_validity)
        val string = SpannableString("$validityLabel ${getValidityDate()}")
        string.setSpan(
            StyleSpan(Typeface.BOLD),
            validityLabel.length,
            string.length,
            Spannable.SPAN_EXCLUSIVE_EXCLUSIVE
        )
        binding.validityHintTv.text = string
    }

    private fun getValidityDate() = SimpleDateFormat(VALIDITY_DATE_FORMAT, Locale.getDefault())
        .format(Calendar.getInstance().apply { add(Calendar.MONTH, MONTHS_TO_BE_VALID) }.time)

    companion object {
        @JvmStatic
        fun destination(docPage: DocPage<ProofOfResidency>, group: Int) = MatiDestination(
            R.id.to_proofOfResidencyHint,
            Bundle().apply {
                putParcelable(ARG_DOC_PAGE, docPage)
                putInt(ARG_DOCUMENT_GROUP, group)
            }
        )

        private const val ARG_DOC_PAGE = "ARG_DOC_PAGE"
        private const val ARG_DOCUMENT_GROUP = "ARG_DOCUMENT_GROUP"

        private const val VALIDITY_DATE_FORMAT = "d MMM, yyyy"
        private const val MONTHS_TO_BE_VALID = -3
    }
}