package com.getmati.mati_sdk.ui.document

import android.os.Bundle
import android.view.View
import android.widget.Toast
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Clicked
import com.getmati.mati_sdk.analytics.events.Selected
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentSelectCountryBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.*
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.Document
import com.getmati.mati_sdk.models.clean.NationalId
import com.getmati.mati_sdk.models.clean.ProofOfResidency
import com.getmati.mati_sdk.models.clean.State
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.doc_hint.DocSkipVm
import com.getmati.mati_sdk.ui.doc_hint.DocumentHintFragment
import com.getmati.mati_sdk.ui.searchable_dialog.SearchableCountryDialog
import com.getmati.mati_sdk.ui.searchable_dialog.SearchableStringDialog
import com.getmati.mati_sdk.ui.utils.loadImageResource
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class SelectCountryFragment : KYCBaseFragment(R.layout.fragment_select_country),
    SearchableStringDialog.SearchableItem,
    SearchableCountryDialog.SearchableItem {

    override val screenName = "documentCountrySelect"

    private val doc by lazy { requireArguments().getParcelable<Document>(ARG_DOCUMENT)!! }
    private val group by lazy { requireArguments().getInt(ARG_DOCUMENT_GROUP) }
    private val skippable by lazy { requireArguments().getBoolean(ARG_SKIPPABLE) }
    private val binding by viewBinding(FragmentSelectCountryBinding::bind)
    private var selectedCountry: Country? = null
    private var selectedRegion: String? = null

    private val docSkipVm: DocSkipVm by viewModel()
    private val countriesRepo by inject<CountriesRepo>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        selectedCountry = doc.country
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setObservers()
    }

    private fun setObservers() {
        viewLifecycleOwner.lifecycleScope.launch {
            docSkipVm.state.collect {
                when (it) {
                    is DocSkipVm.State.Initial -> {
                        setValues()
                    }
                    is DocSkipVm.State.InProgress -> {
                        showProgressBar()
                    }
                    is DocSkipVm.State.SkipSuccess -> {
                        navigation.openNextStep()
                    }
                    is DocSkipVm.State.Error -> {
                        Toast.makeText(
                            requireContext(),
                            getString(R.string.label_something_went_wrong),
                            Toast.LENGTH_LONG
                        ).show()
                        docSkipVm.dropState()
                    }
                }
            }
        }
    }

    private fun setValues(){
        binding.progressBar.visibility = View.INVISIBLE
        binding.actionPrimary.visibility = View.VISIBLE
        setUpSkipTv()
        binding.selectedCountryTvSelectCountryFrag.isEnabled  = true
        binding.selectedRegionTvSelectCountryFrag.isEnabled = true
        binding.innerTitleTvSelectCountryFrag.text = getDocTitle(doc)
        binding.mainIvSelectCountryFrag.loadImageResource(getDocRoundIcon(doc))

        val supportedCountries = countriesRepo.filterSupported(verificationFlow.supportedCountries)
        val countryCode = countriesRepo.getCountryFromInput(verificationVm.initialInputs)?.code
        val allowedRegions = verificationFlow.ipValidation?.allowedRegions ?: emptyList()
        val countries = countriesRepo.reorderWithAllowedRegions(
            source = supportedCountries, countryCode, allowedRegions
        )
        setUpViewComponents(countries)
        setPrimaryButton(selectedCountry?.code, selectedRegion)
    }

    private fun setUpSkipTv() {
        binding.skipTv.visibility = if(skippable) {
            binding.skipTv.setSingleClickListener {
                docSkipVm.skip(doc.id, group)
            }
            View.VISIBLE
        } else{
            View.GONE
        }
    }

    private fun showProgressBar() {
        binding.skipTv.visibility = if (skippable) View.INVISIBLE else View.GONE
        binding.progressBar.visibility = View.VISIBLE
        binding.actionPrimary.visibility = View.INVISIBLE
        binding.selectedCountryTvSelectCountryFrag.isEnabled  = false
        binding.selectedRegionTvSelectCountryFrag.isEnabled = false
        binding.actionPrimary.isEnabled = false
    }

    private fun setupRegions(countryCode: String) {
        if (countriesRepo.shouldShowSelectRegion(countryCode, doc)) {
            binding.selectedRegionTvSelectCountryFrag.visibility =
                View.VISIBLE
            binding.groupSelectRegion.visibility = View.VISIBLE
        } else {
            binding.selectedRegionTvSelectCountryFrag.visibility = View.GONE
            binding.groupSelectRegion.visibility = View.GONE
        }

        if (selectedRegion == null) {
            binding.selectedRegionTvSelectCountryFrag.text =
                when (selectedCountry!!.code) {
                    "US" -> requireContext().getString(R.string.label_select_state_of_document)
                    "CA" -> requireContext().getString(R.string.label_select_province)
                    else -> ""
                }
        } else {
            binding.selectedRegionTvSelectCountryFrag.text = selectedRegion
        }

        binding.selectedRegionTvSelectCountryFrag.setOnClickListener {
            showStatesDialog(when (countryCode) {
                "US" -> State.values().map { it.title }
                "CA" -> Province.values().map { it.title }
                else -> listOf()
            })
        }
    }

    private fun setUpViewComponents(countries: List<Country>) {
        val canChooseCountries = countries.size != 1
        if (selectedCountry != null) {
            if (canChooseCountries) {
                binding.selectedCountryTvSelectCountryFrag.text =
                    selectedCountry?.name
                binding.groupSelectCountry.visibility = View.VISIBLE
            } else {
                binding.innerTitleTvSelectCountryFrag.text = String.format(
                    "%s\n%s",
                    getDocTitle(doc),
                    selectedCountry!!.name
                )
                binding.groupSelectCountry.visibility = View.GONE
            }
            setupRegions(selectedCountry!!.code)
        } else {
            binding.innerTitleTvSelectCountryFrag.text = getDocTitle(doc)
            binding.groupSelectCountry.visibility = View.VISIBLE
            binding.selectedCountryTvSelectCountryFrag.text =
                requireContext().getString(R.string.label_select_country_of_document)
        }
        binding.selectedCountryTvSelectCountryFrag.setOnClickListener {
            if (canChooseCountries) showCountriesDialog(countries)
        }
    }

    private fun setPrimaryButton(
        selectedCountryCode: String?,
        selectedRegion: String?
    ) {
        val isEnabled = selectedCountryCode != null && !(countriesRepo.shouldShowSelectRegion(
            selectedCountryCode,
            doc
        ) && selectedRegion == null)
        binding.actionPrimary.isEnabled = isEnabled
        binding.actionPrimary.alpha =
            if (isEnabled) ALPHA_ENABLED_ACTION_BUTTON else ALPHA_DISABLED_ACTION_BUTTON
        binding.actionPrimary.text =
            requireContext().getString(R.string.label_next)
        binding.actionPrimary.setOnClickListener {
            track(UserAction("primaryButton", Clicked(), screenName))
            if (isEnabled) {
                doc.country = selectedCountry
                doc.region = when (selectedCountryCode) {
                    "US" -> State.codeFromTitle(selectedRegion)
                    "CA" -> Province.codeFromTitle(selectedRegion)
                    else -> null
                }
                navigation.navigateTo(DocumentHintFragment.destination(DocPage(doc, 1), group))
            }
        }
    }

    private fun showStatesDialog(mRegions: List<String>) {
        if (activity == null) return
        val mSearchableListDialog = SearchableStringDialog.newInstance(ArrayList(mRegions))
        if (selectedCountry!!.code == "US") {
            mSearchableListDialog.setTitle(requireContext().getString(R.string.label_select_state_of_document))
        } else if (selectedCountry!!.code == "CA") {
            mSearchableListDialog.setTitle(requireContext().getString(R.string.label_select_province))
        }
        mSearchableListDialog.show(childFragmentManager, "TAG")
    }

    private fun showCountriesDialog(countries: List<Country>) {
        val mSearchableListDialog = SearchableCountryDialog.newInstance(ArrayList(countries))
        mSearchableListDialog.setTitle(requireContext().getString(R.string.label_select_country_of_document))
        mSearchableListDialog.show(childFragmentManager, "TAG")
    }

    companion object {
        private const val ARG_DOCUMENT = "ARG_DOCUMENT"
        private const val ARG_DOCUMENT_GROUP = "ARG_DOCUMENT_GROUP"
        private const val ARG_SKIPPABLE = "ARG_SKIPPABLE"

        private const val ALPHA_DISABLED_ACTION_BUTTON = 0.5f
        private const val ALPHA_ENABLED_ACTION_BUTTON = 1f

        fun destination(doc: Document, documentGroup: Int, skippable:Boolean = false): MatiDestination {
            return MatiDestination(R.id.to_selectCountry, Bundle().apply {
                putParcelable(ARG_DOCUMENT, doc)
                putInt(ARG_DOCUMENT_GROUP, documentGroup)
                putBoolean(ARG_SKIPPABLE, skippable)
            })
        }
    }

    override fun onSearchableItemClicked(item: Country) {
        if (selectedCountry != item) {
            selectedRegion = null
            selectedCountry = item
            track(UserAction("selectCountry", Selected(item.code), screenName))
            binding.selectedCountryTvSelectCountryFrag.text = item.name
            setupRegions(item.code)
            setPrimaryButton(selectedCountry?.code, selectedRegion)
        }
    }

    override fun onSearchableItemClicked(item: String) {
        if (selectedRegion != item) {
            selectedRegion = item
            track(UserAction("selectRegion", Selected(item), screenName))
            binding.selectedRegionTvSelectCountryFrag.text = selectedRegion
            setPrimaryButton(selectedCountry?.code, selectedRegion)
        }
    }
}

internal fun KYCBaseFragment.getDocRoundIcon(doc: Document) = when (doc) {
    is CustomDoc -> R.drawable.ic_custom_doc_oval
    is Passport -> R.drawable.ic_passport_oval
    is NationalId -> R.drawable.ic_national_id_oval
    is DrivingLicense -> R.drawable.ic_driving_license_oval
    is ProofOfResidency -> R.drawable.ic_proof_of_address_oval
    else -> throw IllegalArgumentException("Unhandled document")
}

internal fun KYCBaseFragment.getDocTitle(doc: Document) = when (doc) {
    is CustomDoc -> doc.name
    is Passport -> getString(R.string.label_passport)
    is NationalId -> getString(R.string.label_national_id)
    is DrivingLicense -> getString(R.string.label_drivers_license)
    is ProofOfResidency -> getString(R.string.label_proof_of_address)
    else -> throw IllegalArgumentException("Unhandled document")
}