package com.getmati.mati_sdk.ui.document

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import android.widget.Toast
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.analytics.events.Selected
import com.getmati.mati_sdk.analytics.events.UserAction
import com.getmati.mati_sdk.analytics.track
import com.getmati.mati_sdk.databinding.FragmentSelectDocumentBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.Country
import com.getmati.mati_sdk.models.clean.CustomDoc
import com.getmati.mati_sdk.models.clean.DocPage
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.DocumentVerificationStep
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.doc_hint.DocSkipVm
import com.getmati.mati_sdk.ui.doc_hint.DocumentHintFragment
import com.getmati.mati_sdk.ui.searchable_dialog.SearchableCountryDialog
import com.getmati.mati_sdk.ui.utils.findView
import com.getmati.mati_sdk.ui.utils.loadImageResource
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.android.ext.android.inject
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class SelectDocumentFragment : KYCBaseFragment(R.layout.fragment_select_document),
    SearchableCountryDialog.SearchableItem {

    override val screenName = "documentSelect"
    private val binding by viewBinding(FragmentSelectDocumentBinding::bind)

    private val step by lazy {
        requireArguments().getParcelable<DocumentVerificationStep>(
            ARG_DOCUMENT_VERIFICATION_STEP
        )!!
    }
    
    private var skippable = true

    private val docSkipVm: DocSkipVm by viewModel()
    private val countriesRepo by inject<CountriesRepo>()

    private var selectedCountry: Country? = null
    
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        defineSkippable()
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        defineSkippable()
        setObservers()
    }

    override fun onSearchableItemClicked(item: Country) {
        track(UserAction("selectCountry", Selected(item.code), screenName))
        setSelectedCountry(item)
    }

    private fun defineSkippable() {
        for (doc in step.acceptableDocuments) {
            skippable = if(doc is CustomDoc) skippable && doc.skippable else false
        }
    }

    private fun setObservers() {
        viewLifecycleOwner.lifecycleScope.launch {
            docSkipVm.state.collect {
                when (it) {
                    is DocSkipVm.State.Initial -> {
                        setValues()
                    }
                    is DocSkipVm.State.InProgress -> {
                        showProgressBar()
                    }
                    is DocSkipVm.State.SkipSuccess -> {
                        navigation.openNextStep()
                    }
                    is DocSkipVm.State.Error -> {
                        Toast.makeText(
                            requireContext(),
                            getString(R.string.label_something_went_wrong),
                            Toast.LENGTH_LONG
                        ).show()
                        docSkipVm.dropState()
                    }
                }
            }
        }
    }

    private fun setValues(){
        binding.outerTitleTvStartVerificationFrag.text = requireContext().getString(R.string.label_select_document)
        binding.progressBar.visibility = View.INVISIBLE
        setUpDocuments()
        setUpSkipTv()
        setUpCountriesTv()
    }

    private fun showProgressBar() {
        binding.skipTv.visibility = if (skippable) View.INVISIBLE else View.GONE
        binding.progressBar.visibility = View.VISIBLE
        setDocsSelectable(false)
        binding.selectedCountryTvSelectDocumentFrag.isEnabled = false
    }

    private fun setUpDocuments() {
        for (doc in step.acceptableDocuments) {
            val itemView = LayoutInflater.from(context)
                .inflate(R.layout.item_document, binding.docsContainerLlSelectDocumentFrag, false)
            val titleTV = itemView.findViewById<TextView>(R.id.title_tv_document_item)
            val mainIV = itemView.findViewById<ImageView>(R.id.main_iv_item_document)
            titleTV.text = getDocTitle(doc)
            mainIV.loadImageResource(getDocRoundIcon(doc))
            itemView.setOnClickListener {
                selectedCountry?.let {
                    track(UserAction("selectDocument", Selected(doc.id), screenName))
                    doc.country = it
                    if (countriesRepo.shouldShowSelectRegion(it.code, doc)) {
                        navigation.navigateTo(SelectCountryFragment.destination(doc, step.group))
                    } else {
                        navigation.navigateTo(
                            DocumentHintFragment.destination(DocPage(doc, 1), step.group)
                        )
                    }
                }
            }
            binding.docsContainerLlSelectDocumentFrag.addView(itemView)
        }
    }

    private fun setUpSkipTv() {
        binding.skipTv.visibility = if(skippable) {
            binding.skipTv.setSingleClickListener {
                docSkipVm.skip(step.acceptableDocuments.first().id, step.group)
            }
            View.VISIBLE
        } else{
            View.GONE
        }
    }

    private fun setUpCountriesTv() {
        val supportedCountries = countriesRepo.filterSupported(verificationFlow.supportedCountries)
        val countryCode = countriesRepo.getCountryFromInput(verificationVm.initialInputs)?.code
        val allowedRegions = verificationFlow.ipValidation?.allowedRegions ?: emptyList()
        val countries = countriesRepo.reorderWithAllowedRegions(
            source = supportedCountries, countryCode, allowedRegions
        )
        if (countries.size == 1) {
            val country = countries.first()
            setSelectedCountry(country)
            binding.labelCountry.text =
                "${requireContext().getString(R.string.label_country)}: ${country?.name}"
            binding.selectedCountryTvSelectDocumentFrag.visibility = View.GONE
            binding.selectedCountryTvSelectDocumentFrag.text = country.name
        } else {
            binding.selectedCountryTvSelectDocumentFrag.visibility = View.VISIBLE
        }

        if (selectedCountry == null) {
            binding.selectedCountryTvSelectDocumentFrag.text = requireContext().getString(R.string.label_select_country)
            setDocsSelectable(false)
        } else {
            binding.selectedCountryTvSelectDocumentFrag.text = selectedCountry?.name
            setDocsSelectable(true)
        }
        binding.selectedCountryTvSelectDocumentFrag.isEnabled = true
        binding.selectedCountryTvSelectDocumentFrag.setOnClickListener {
            showCountriesDialog(countries)
        }
    }

    private fun showCountriesDialog(countries: List<Country>) {
        if (activity == null || countries.isEmpty()) return
        val mSearchableListDialog = SearchableCountryDialog.newInstance(ArrayList(countries))
        mSearchableListDialog.setTitle(requireContext().getString(R.string.label_select_country))
        mSearchableListDialog.show(childFragmentManager, "TAG")
    }

    private fun setSelectedCountry(country: Country) {
        selectedCountry = country
        findView<TextView>(R.id.selected_country_tv_select_document_frag).text = country.name
        setDocsSelectable(true)
    }

    private fun setDocsSelectable(selectable: Boolean) {
        for (i in 0 until findView<ViewGroup>(R.id.docs_container_ll_select_document_frag).childCount) {
            findView<ViewGroup>(R.id.docs_container_ll_select_document_frag).getChildAt(i).isEnabled =
                selectable
        }
        findView<ViewGroup>(R.id.docs_container_ll_select_document_frag).alpha =
            if (selectable) ALPHA_ENABLED_ACTION_BUTTON else ALPHA_DISABLED_ACTION_BUTTON
    }

    companion object {
        private const val ARG_DOCUMENT_VERIFICATION_STEP = "ARG_DOCUMENT_VERIFICATION_STEP"

        private const val ALPHA_DISABLED_ACTION_BUTTON = 0.3f
        private const val ALPHA_ENABLED_ACTION_BUTTON = 1f
        fun destination(step: DocumentVerificationStep): MatiDestination {
            return MatiDestination(R.id.to_selectDocument, Bundle().apply {
                putParcelable(ARG_DOCUMENT_VERIFICATION_STEP, step)
            })
        }
    }
}