package com.getmati.mati_sdk.ui.email.email_submission

import android.os.Bundle
import android.view.View
import android.view.inputmethod.EditorInfo
import androidx.core.widget.addTextChangedListener
import androidx.fragment.app.activityViewModels
import androidx.lifecycle.lifecycleScope
import com.getmati.mati_sdk.R
import com.getmati.mati_sdk.databinding.FragmentEmailSubmissionBinding
import com.getmati.mati_sdk.mati_navigation.MatiDestination
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.setSingleClickListener
import com.getmati.mati_sdk.ui.common.KYCBaseFragment
import com.getmati.mati_sdk.ui.email.EmailSharedVm
import com.getmati.mati_sdk.ui.email.EmailSharedVmFactory
import com.getmati.mati_sdk.ui.email.email_validation.EmailVerificationFragment
import com.getmati.mati_sdk.ui.email.email_validation.trackCoolDown
import com.getmati.mati_sdk.ui.error.BaseErrorFragment
import com.getmati.mati_sdk.ui.error.prepareMediaErrorScreenData
import com.getmati.mati_sdk.ui.utils.view_binding.viewBinding
import com.getmati.mati_sdk.widgets.hideKeyboard
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.launch
import org.koin.androidx.viewmodel.ext.android.viewModel

internal class EmailSubmissionFragment : KYCBaseFragment(R.layout.fragment_email_submission) {

    override val screenName get() = "emailSubmission"

    private val isOptional by lazy { requireArguments().getBoolean(ARG_IS_OPTIONAL) }
    private val attemptLimit by lazy { requireArguments().getInt(ARG_ATTEMPT_LIMIT) }
    private val coolDown by lazy { requireArguments().getInt(ARG_COOL_DOWN) }
    private val binding by viewBinding(FragmentEmailSubmissionBinding::bind)

    private var coolDownCounter: Job? = null

    private val emailSubmissionVm by viewModel<EmailSubmissionVm>()

    private val emailSharedVm: EmailSharedVm by activityViewModels {
        EmailSharedVmFactory(verificationActivity!!, requireArguments())
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        setListeners()
        observeState()
    }

    private fun setListeners() {
        binding.skipTv.setSingleClickListener {
            emailSubmissionVm.skip()
        }
        binding.emailEt.addTextChangedListener {
            if (emailSubmissionVm.state.value is EmailSubmissionVm.State.Initial)
                showInitialState()
            emailSubmissionVm.dropState()
        }
        binding.emailEt.setOnEditorActionListener { v, actionId, event ->
            if (actionId == EditorInfo.IME_ACTION_DONE) {
                v.hideKeyboard()
                binding.actionPrimary.performClick()
                return@setOnEditorActionListener true
            }
            return@setOnEditorActionListener false
        }
    }

    private fun observeState() {
        viewLifecycleOwner.lifecycleScope.launch(Dispatchers.Main) {
            emailSubmissionVm.state.collect {
                when (it) {
                    is EmailSubmissionVm.State.Initial -> {
                        showInitialState()
                    }
                    is EmailSubmissionVm.State.InvalidEmail -> {
                        showError(getString(MediaVerificationError.INVALID_EMAIL.subtitle))
                    }
                    is EmailSubmissionVm.State.InProgress -> {
                        showLoadingState()
                    }
                    is EmailSubmissionVm.State.Error -> {
                        showRequestFailureState(it.error.type)
                    }
                    is EmailSubmissionVm.State.Success -> {
                        onEmailSubmitted(it.email)
                    }
                    is EmailSubmissionVm.State.SkipSuccess -> {
                        navigation.openNextStep()
                    }
                }
            }
        }
    }

    private fun showInitialState() {
        binding.progressBar.visibility = View.INVISIBLE
        binding.skipTv.visibility = if (isOptional) View.VISIBLE else View.GONE

        showResend()
        if (emailSharedVm.resendLocked) {
            binding.invalidEmailErrorTv.text =
                getString(MediaVerificationError.EMAIL_TOO_MANY_RESENDS.subtitle)
            binding.invalidEmailErrorTv.visibility = View.VISIBLE
            binding.emailEt.isEnabled = false
        } else {
            binding.invalidEmailErrorTv.visibility = View.INVISIBLE
            binding.emailEt.isEnabled = true
        }
    }

    private fun showLoadingState() {
        binding.invalidEmailErrorTv.visibility = View.INVISIBLE
        binding.progressBar.visibility = View.VISIBLE
        binding.skipTv.visibility = if (isOptional) View.INVISIBLE else View.GONE
        binding.actionPrimary.visibility = View.INVISIBLE
        binding.resendInfoTv.visibility = View.INVISIBLE
        binding.emailEt.isEnabled = false
    }

    private fun showError(message: String) {
        showInitialState()
        binding.invalidEmailErrorTv.text = message
        binding.invalidEmailErrorTv.visibility = View.VISIBLE
    }

    private fun showRequestFailureState(error: MediaVerificationError) {
        when (error) {
            MediaVerificationError.INVALID_EMAIL -> showError(getString(error.subtitle))
            MediaVerificationError.INPUT_LOCKED -> navigation.openNextStep()
            MediaVerificationError.EMAIL_TOO_MANY_RESENDS -> {
                showError(getString(error.subtitle))
            }
            else -> {
                emailSubmissionVm.dropState()
                navigation.navigateTo(
                    BaseErrorFragment.destination(
                        prepareMediaErrorScreenData(
                            title = getString(error.title),
                            subHeading = getString(error.subtitle),
                            primaryCTALabel = getString(R.string.label_try_again)
                        )
                    )
                )
            }
        }
    }

    private fun onEmailSubmitted(email: String) {
        emailSharedVm.onNewEmailSubmitted(email)
        emailSubmissionVm.dropState()
        navigation.navigateTo(EmailVerificationFragment.destination(email, attemptLimit, coolDown))
    }

    private fun showResend() {
        binding.actionPrimary.visibility = View.VISIBLE
        binding.invalidEmailErrorTv.visibility = View.INVISIBLE
        val lastEmail = emailSharedVm.lastEmail
        val sameEmail = lastEmail == binding.emailEt.text.toString()
        if (emailSharedVm.resendLocked) {
            coolDownCounter?.cancel()
            coolDownCounter = null

            binding.resendInfoTv.visibility = View.INVISIBLE

            binding.actionPrimary.text = getString(R.string.label_enter_code)
            binding.actionPrimary.setSingleClickListener {
                navigation.navigateTo(
                    EmailVerificationFragment.destination(
                        lastEmail!!,
                        attemptLimit,
                        coolDown
                    )
                )
            }
        } else if (emailSharedVm.resendIsAllowed()) {
            coolDownCounter?.cancel()
            coolDownCounter = null

            if (lastEmail != null && sameEmail) {
                binding.resendInfoTv.visibility = View.VISIBLE
                binding.resendInfoTv.underline = true
                binding.resendInfoTv.text = getString(R.string.label_resend)
                binding.resendInfoTv.setSingleClickListener {
                    emailSubmissionVm.process(lastEmail)
                }
            } else {
                binding.resendInfoTv.visibility = View.INVISIBLE
            }

            binding.actionPrimary.text = if (lastEmail == binding.emailEt.text.toString()) {
                getString(R.string.label_enter_code)
            } else {
                getString(R.string.primary_action_title_email_submission_fragment)
            }
            binding.actionPrimary.setSingleClickListener {
                if (lastEmail == binding.emailEt.text.toString()) {
                    navigation.navigateTo(
                        EmailVerificationFragment.destination(
                            lastEmail,
                            attemptLimit,
                            coolDown
                        )
                    )
                } else {
                    emailSubmissionVm.process(binding.emailEt.text.toString())
                }
            }
        } else {
            binding.resendInfoTv.visibility = View.VISIBLE
            binding.resendInfoTv.setOnClickListener(null)
            binding.resendInfoTv.underline = false

            binding.actionPrimary.run {
                text = if (sameEmail) {
                    getString(R.string.label_enter_code)
                } else {
                    getString(R.string.primary_action_title_email_submission_fragment)
                }
                setSingleClickListener {
                    if (sameEmail) {
                        navigation.navigateTo(
                            EmailVerificationFragment.destination(
                                lastEmail!!,
                                attemptLimit,
                                coolDown
                            )
                        )
                    } else {
                        binding.invalidEmailErrorTv.visibility = View.VISIBLE
                        binding.invalidEmailErrorTv.setText(R.string.email_error_resend_locked)
                    }
                }
            }

            if (coolDownCounter?.isActive == true) return
            coolDownCounter = viewLifecycleOwner.lifecycleScope.launchWhenStarted {
                trackCoolDown(
                    coolDownUntil = emailSharedVm.coolDownUntil,
                    onEachSecond = { time ->
                        binding.resendInfoTv.text = String.format(
                            "%s, %s",
                            getString(R.string.label_sms_check_resend),
                            time
                        )
                    },
                    onFinish = { if (binding.actionPrimary.visibility == View.VISIBLE) showResend() }
                )
            }
        }
    }

    companion object {
        private const val ARG_IS_OPTIONAL = "ARG_IS_OPTIONAL"
        private const val ARG_ATTEMPT_LIMIT = "ARG_ATTEMPT_LIMIT"
        private const val ARG_COOL_DOWN = "ARG_COOL_DOWN"

        fun destination(isOptional: Boolean, attemptLimit: Int, coolDown: Int): MatiDestination {
            return MatiDestination(R.id.to_email_submission, Bundle().apply {
                putBoolean(ARG_IS_OPTIONAL, isOptional)
                putInt(ARG_ATTEMPT_LIMIT, attemptLimit)
                putInt(ARG_COOL_DOWN, coolDown)
            })
        }
    }
}