package com.getmati.mati_sdk.ui.email.email_submission

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.getmati.mati_sdk.isValidEmail
import com.getmati.mati_sdk.models.clean.MediaVerificationError
import com.getmati.mati_sdk.server.RequestKit
import com.getmati.mati_sdk.server.verification.VerificationError
import com.getmati.mati_sdk.ui.email.EmailVerificationRepository
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

internal class EmailSubmissionVm(private val emailVerificationRepository: EmailVerificationRepository) : ViewModel() {

    private val _state = MutableStateFlow<State>(State.Initial)
    val state: StateFlow<State> get() = _state

    fun process(email: String) {
        if (email.isValidEmail()) {
            submitEmail(email)
        } else {
            _state.value = State.InvalidEmail
        }
    }

    private fun submitEmail(email: String) {
        _state.value = State.InProgress
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = emailVerificationRepository.submitEmail(email).run {
                withContext(Dispatchers.Main) {
                    if (error == null) {
                        State.Success(data!!)
                    } else {
                        State.Error(error)
                    }
                }
            }
        }
    }


    fun dropState() {
        _state.value = State.Initial
    }

    fun skip() {
        _state.value = State.InProgress
        viewModelScope.launch(Dispatchers.IO) {
            _state.value = emailVerificationRepository.skip().run {
                withContext(Dispatchers.Main) {
                    if (error == null || error.type == MediaVerificationError.INPUT_SKIPPED) {
                        State.SkipSuccess
                    } else {
                        State.Error(error)
                    }
                }
            }
        }
    }

    sealed class State {
        object Initial : State()
        object InvalidEmail : State()
        object InProgress : State()
        data class Success(val email: String) : State()
        object SkipSuccess : State()
        data class Error(val error: VerificationError) : State()
    }
}